# run_skeleton_task.py (Completed)

import math
import time
import numpy as np
from typing import Dict, Tuple, Optional

from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# NOTE: every primitive-level skill has already been implemented in skill_code
#       We simply import * so we can call them directly.
from skill_code import *           # noqa: F401, F403

from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

from object_positions import get_object_positions


# --------------------------------------------------------------------------- #
# ----------------------------  Helper Utilities  --------------------------- #
# --------------------------------------------------------------------------- #
def quaternion_from_euler(rx: float, ry: float, rz: float) -> np.ndarray:
    """
    Convert Euler angles (radians, XYZ-extrinsic / ZYX-intrinsic) to quaternion
    in xyzw order.  Only standard library math is used.
    """
    cx, cy, cz = math.cos(rx / 2.0), math.cos(ry / 2.0), math.cos(rz / 2.0)
    sx, sy, sz = math.sin(rx / 2.0), math.sin(ry / 2.0), math.sin(rz / 2.0)

    qw = (cx * cy * cz + sx * sy * sz)
    qx = (sx * cy * cz - cx * sy * sz)
    qy = (cx * sy * cz + sx * cy * sz)
    qz = (cx * cy * sz - sx * sy * cz)
    return np.asarray([qx, qy, qz, qw], dtype=np.float32)


def find_drawer_handle_pose(positions: Dict[str, Tuple[float, float, float]]
                            ) -> Optional[np.ndarray]:
    """
    Try to locate a drawer-handle (or drawer knob) in the provided dictionary.
    The dictionary keys are assumed to be object names coming from the
    RLBench scene.  We return the position as a NumPy array if found, else None.
    """
    # Very common RLBench drawer tasks name their handles ‘drawer_handle’
    # or contain the word ‘handle’.  We simply look for that.
    for name, pos in positions.items():
        lname = name.lower()
        if ("handle" in lname or "knob" in lname or "drawer" in lname) and \
           ("handle" in lname or "knob" in lname):
            return np.asarray(pos, dtype=np.float32)
    return None


# --------------------------------------------------------------------------- #
# ----------------------------  Main Task Runner  --------------------------- #
# --------------------------------------------------------------------------- #
def run_skeleton_task() -> None:
    """
    Generic runner that demonstrates:
      • Environment set-up / tear-down
      • Basic exploration to establish the ‘rotated’ predicate
      • High-level plan for opening a drawer (rotate → move → pull)
    The code is deliberately robust to missing objects or premature task
    termination.  All actions come from the pre-implemented skill library.
    """
    print("===== Starting Skeleton Task =====")

    # === 1)  Environment Setup  =================================================
    env, task = setup_environment()

    try:
        # ------------------------------------------------------------------ #
        # Initial (domain-agnostic) reset & optional video initialisation
        # ------------------------------------------------------------------ #
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap task.step & get_observation so that any call is automatically
        # recorded to video.  The helper wrappers are provided by video.py.
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------ #
        # 2)  Gather low-level state information (object positions, etc.)
        # ------------------------------------------------------------------ #
        positions: Dict[str, Tuple[float, float, float]] = get_object_positions()
        handle_pos = find_drawer_handle_pose(positions)

        if handle_pos is None:
            # We could not detect a drawer handle—gracefully terminate.
            print("[Error] No drawer handle could be located in the scene.")
            return

        print(f"[Info] Detected drawer handle at world position: {handle_pos}")

        # ------------------------------------------------------------------ #
        # 3)  Exploration Phase – establishing the ‘rotated’ predicate
        # ------------------------------------------------------------------ #
        #    The feedback we have received indicates that the predicate
        #    (rotated ?g ninety_deg) was missing in earlier plans.  In the
        #    RLBench environment that manifests as the gripper starting in an
        #    arbitrary orientation.  We therefore *always* rotate the end-effector
        #    so that the fingers are perpendicular to the drawer front
        #    (i.e. rotated 90° around the local Z axis).
        #
        #    We treat this rotation as our domain-level “exploration” of that
        #    predicate: once the rotate skill finishes successfully, the
        #    predicate (rotated gripper ninety_deg) is now *true* in the
        #    physical simulation and subsequent high-level actions that depend
        #    on it (move-to-side, pull, …) can be executed without issue.
        # ------------------------------------------------------------------ #
        target_quat = quaternion_from_euler(0.0, 0.0, math.pi / 2.0)   # 90° about Z
        print("[Exploration] Executing rotate() to satisfy missing 'rotated' predicate.")
        try:
            # rotate skill returns (obs, reward, done); we only need the last two
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=120,
                threshold=0.03,
                timeout=15.0,
            )
            if done:
                print("[Early Termination] Task reported done during exploration.")
                return
        except Exception as e:
            print(f"[Warning] rotate() skill threw an exception: {e}. Continuing.")

        # ------------------------------------------------------------------ #
        # 4)  High-level Plan – Move close to the handle & Pull drawer
        # ------------------------------------------------------------------ #
        # Because different RLBench tasks use different coordinate frames we
        # adopt the following *very* conservative strategy:
        #   a) Approach above the handle ( +0.10 m in Z )                – move()
        #   b) Descend *onto* the handle ( –0.09 m in Z )                – move()
        #   c) Pull along the negative Y axis (typical drawer direction) – pull()
        #
        # If your RLBench task is oriented differently this will still succeed
        # because the pull() skill itself handles small local adjustments.
        # ------------------------------------------------------------------ #
        approach_height = 0.10     # 10 cm above the handle
        descend_height  = 0.09     # 9 cm down to touch the handle
        pull_distance   = 0.15     # 15 cm backward pull

        # ---- a)  Approach above the handle  -----------------------------
        above_handle = handle_pos.copy()
        above_handle[2] += approach_height
        print(f"[Plan] move() – approaching above handle: {above_handle}")
        obs, reward, done = move(
            env,
            task,
            target_pos=above_handle,
            approach_distance=0.15,
            max_steps=120,
            threshold=0.01,
            approach_axis="z",
            timeout=15.0,
        )
        if done:
            print("[Early Termination] Task ended after first move()!")
            return

        # ---- b)  Descend onto the handle  -------------------------------
        on_handle = handle_pos.copy()
        on_handle[2] += (approach_height - descend_height)
        print(f"[Plan] move() – descending onto handle: {on_handle}")
        obs, reward, done = move(
            env,
            task,
            target_pos=on_handle,
            approach_distance=0.02,
            max_steps=60,
            threshold=0.005,
            approach_axis="z",
            timeout=10.0,
        )
        if done:
            print("[Early Termination] Task ended after second move()!")
            return

        # ---- c)  Pull drawer open  --------------------------------------
        # pull() typically performs a straight-line motion.  We provide a
        # *direction* vector rather than an absolute target so the skill can
        # decide how far it should travel before satisfying its internal stop
        # conditions.  If the skill implementation you have takes explicit
        # start/end poses instead, adapt the call-signature accordingly.
        print("[Plan] pull() – opening the drawer.")
        obs, reward, done = pull(
            env,
            task,
            direction=np.asarray([0.0, -1.0, 0.0], dtype=np.float32),
            distance=pull_distance,
            max_steps=150,
            threshold=0.01,
            timeout=20.0,
        )
        if done:
            print("[Success] Drawer opened.  Task finished!")
        else:
            print("[Info] pull() completed but task did not signal done.")

    finally:
        # === 5)  Clean-up =======================================================
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


# Standard Python entry-point guard
if __name__ == "__main__":
    run_skeleton_task()