# run_skeleton_task.py

import time
import numpy as np
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
# ── Pre-implemented skills (do **not** redefine) ──
from skill_code import move, pick, place, rotate, pull


# ──────────────────────────────────────────────────────────────────────────────
# Helper utilities (no new low-level skills are introduced here)
# ──────────────────────────────────────────────────────────────────────────────
SAFE_FORCE_RANGE = (0.0, 40.0)      # (N, N) – adjust to your hardware limits
GRIPPER_OPEN_VALUE = 1.0            # what the observation returns for “open”
GRIPPER_CLOSED_VALUE = -1.0         # idem for “closed”


def _within_safe_force(obs) -> bool:
    """Return True iff the gripper force in the observation is safe."""
    # Many RLBench tasks expose `obs.gripper_force`; fall back to True if absent
    force = getattr(obs, "gripper_force", (SAFE_FORCE_RANGE[0] + SAFE_FORCE_RANGE[1]) / 2)
    return SAFE_FORCE_RANGE[0] <= force <= SAFE_FORCE_RANGE[1]


def _gripper_is_open(obs) -> bool:
    """Heuristic check for open gripper based on openness sign."""
    return getattr(obs, "gripper_openness", GRIPPER_OPEN_VALUE) > 0.0


def _approach_above(position, dz=0.12):
    """Return a point dz metres above a given XYZ position."""
    return np.asarray(position) + np.array([0.0, 0.0, dz])


def _attempt_pick(env, task, name, pos):
    """Generic pick sequence with validations and safety checks."""
    print(f"[Exploration] → Trying to pick “{name}”")
    # 1) Move above the object
    obs, reward, done = move(env, task, _approach_above(pos))
    if done:
        return done
    # 2) Safety check: gripper status & force
    if not _within_safe_force(obs):
        print(f"[Warning] Unsafe gripper force before picking {name}. Skipping.")
        return False
    if not _gripper_is_open(obs):
        # If the predefined pick skill implicitly closes, we first make sure it
        # starts from an open configuration by opening then re-moving
        print(f"[Info] Gripper not open – re-opening by calling place(no-obj).")
        # “place” with dummy coordinates acts like an “open-gripper” in many skill
        # sets – we pass the current TCP position
        place(env, task, target_pos=obs.gripper_pose[:3])
    # 3) Lower close to the object
    obs, reward, done = move(env, task, np.asarray(pos) + np.array([0.0, 0.0, 0.02]))
    if done:
        return done
    # 4) Pick
    try:
        obs, reward, done = pick(env,
                                 task,
                                 target_pos=pos,
                                 approach_distance=0.03,
                                 threshold=0.005,
                                 timeout=5.0)
    except Exception as exc:
        print(f"[Error] Pick skill raised while handling “{name}”: {exc}")
        done = False
    return done


def _attempt_place_in_bin(env, task, bin_pos):
    """Drop whatever is held above the disposal bin."""
    print(f"[Exploration] → Placing in bin at {bin_pos}")
    obs, reward, done = move(env, task, _approach_above(bin_pos))
    if done:
        return done
    obs, reward, done = move(env, task, np.asarray(bin_pos) + np.array([0.0, 0.0, 0.05]))
    if done:
        return done
    try:
        obs, reward, done = place(env,
                                  task,
                                  target_pos=np.asarray(bin_pos) + np.array([0.0, 0.0, 0.02]),
                                  approach_distance=0.02,
                                  threshold=0.005,
                                  timeout=5.0)
    except Exception as exc:
        print(f"[Error] Place skill failed: {exc}")
        done = False
    return done


# ──────────────────────────────────────────────────────────────────────────────
# Main routine
# ──────────────────────────────────────────────────────────────────────────────
def run_skeleton_task():
    print("═══════════  Running Skeleton Task  ═══════════")
    env, task = setup_environment()
    try:
        # Reset, obtain initial observation and optionally enable video
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve world-state – all known objects and their positions
        positions = get_object_positions()          # dict: name → (x, y, z)
        if not positions:
            print("[Error] object_positions returned an empty dict – aborting.")
            return

        # Optional: locate a disposal bin or table; fall back to first object + offset
        disposal_bin = None
        for n in positions.keys():
            if "bin" in n or "trash" in n:
                disposal_bin = positions[n]
                break
        if disposal_bin is None:
            # default location far in +Y direction
            sample = next(iter(positions.values()))
            disposal_bin = np.asarray(sample) + np.array([0.0, 0.3, 0.0])
            print(f"[Info] Generated synthetic bin position at {disposal_bin}")

        # -------------------------------------------
        #  Exploration loop to discover missing info
        # -------------------------------------------
        # The goal here is not task-specific success but to interact with every
        # visible object so that latent predicates (weight-known, durability-known,
        # lock-known, …) become true in the symbolic state.
        for name, pos in positions.items():
            # Validation (#1 from feedback)
            if pos is None or len(pos) != 3:
                print(f"[Warning] Malformed position for «{name}». Skipping.")
                continue

            # Try a pick-and-place cycle
            done = _attempt_pick(env, task, name, pos)
            if done:
                print("[Task] Environment signalled episode completion mid-pick.")
                break

            # Place in bin so that handempty becomes true again
            done = _attempt_place_in_bin(env, task, disposal_bin)
            if done:
                print("[Task] Environment signalled episode completion mid-place.")
                break

            # ‑- Small pause for stability
            for _ in range(3):
                obs, reward, done = task.step(np.zeros(env.action_shape))
                if done:
                    break
            if done:
                break

        # -------------------------------------------
        # Optionally deal with drawers (rotate, pull)
        # -------------------------------------------
        for name in positions.keys():
            if "drawer" not in name.lower():
                continue
            print(f"[Drawer] Attempting to open «{name}»")
            handle_pos = positions[name]
            # Approach handle, rotate wrist, grasp, pull …
            # Only high-level calls because low-level logic lives in skills
            obs, reward, done = move(env, task, _approach_above(handle_pos))
            if done:
                break
            # Minor fine-tuning
            obs, reward, done = move(env, task, handle_pos + np.array([0.0, 0.0, 0.02]))
            if done:
                break
            # The ready-made “pull” skill is assumed to align & execute
            try:
                obs, reward, done = pull(env, task)
            except Exception as exc:
                print(f"[Drawer] Pull failed on {name}: {exc}")
                done = False
            if done:
                break

        print("═══════════  Exploration phase complete  ═══════════")
    finally:
        shutdown_environment(env)
    print("═══════════  End of Skeleton Task  ═══════════")


if __name__ == "__main__":
    run_skeleton_task()