# run_skeleton_task.py (Completed Executable)

import inspect
import time
import numpy as np

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull   # do NOT redefine these
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# --------------------------------------------------------------------------
#  Utility: very light-weight PDDL parser (only grabs names in (:predicates)
# --------------------------------------------------------------------------
def _extract_predicates(pddl_str: str):
    preds, grab = set(), False
    for raw in pddl_str.splitlines():
        line = raw.strip()
        if not line:
            continue
        if line.startswith('(:predicates'):
            grab = True
            continue
        if grab:
            if line.startswith(')'):
                break
            # remove comments & parens, then split
            tokens = line.replace('(', ' ').replace(')', ' ').split()
            if tokens:
                preds.add(tokens[0])
    return preds


# --------------------------------------------------------------------------
#  We bundle the two domain snippets that were given in the prompt so we can
#  automatically detect a predicate that appears in the exploration domain
#  but is missing from the main (combined-domain) description.
# --------------------------------------------------------------------------
_COMBINED_DOMAIN_PDDL = """
(define (domain combined-domain)
  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)
  (:types object location drawer gripper position angle)
  (:predicates
    (at ?obj - object ?loc - location)
    (holding ?obj - object)
    (handempty)
    (is-locked ?d - drawer)
    (is-open ?d - drawer)
    (rotated ?g - gripper ?a - angle)
    (gripper-at ?g - gripper ?p - position)
    (holding-drawer ?g - gripper ?d - drawer)
    (is-side-pos ?p - position ?d - drawer)
    (is-anchor-pos ?p - position ?d - drawer)
  )
)
"""

_EXPLORATION_DOMAIN_PDDL = """
(define (domain exploration)
  (:requirements :strips :typing :conditional-effects :universal-preconditions)
  (:types robot object location)
  (:predicates
    (robot-at ?r - robot ?loc - location)
    (at ?obj - object ?loc - location)
    (identified ?obj - object)
    (temperature-known ?obj - object)
    (holding ?obj - object)
    (handempty)
    (weight-known ?obj - object)
    (durability-known ?obj - object)
    (lock-known ?obj - object)
  )
)
"""


def _detect_missing_predicates():
    """Returns the set of predicates that are in exploration PDDL but not in the main domain."""
    base = _extract_predicates(_COMBINED_DOMAIN_PDDL)
    exp  = _extract_predicates(_EXPLORATION_DOMAIN_PDDL)
    return sorted(list(exp - base))


# --------------------------------------------------------------------------
#  Generic wrapper that calls a skill *if* its signature matches what we have
#  available.  This prevents crashes when the exact skill signature differs
#  across tasks.
# --------------------------------------------------------------------------
def _flex_call(func, *args, **kwargs):
    sig = inspect.signature(func)
    bound = {}
    # positional mapping
    for pos, (name, param) in enumerate(sig.parameters.items()):
        if pos < len(args):
            bound[name] = args[pos]
    # kw mapping
    for k, v in kwargs.items():
        if k in sig.parameters:
            bound[k] = v
    # fill with defaults if necessary / possible
    for name, param in sig.parameters.items():
        if name not in bound and param.default is not inspect._empty:
            bound[name] = param.default
    # re-order according to the function’s real signature
    ordered = [bound[n] for n in sig.parameters]
    return func(*ordered)


# --------------------------------------------------------------------------
#  MAIN ENTRY
# --------------------------------------------------------------------------
def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    # ----------------------------------------------------
    # 1) Boot the RLBench environment
    # ----------------------------------------------------
    env, task = setup_environment()

    try:
        descriptions, obs = task.reset()

        # ------------- Video / Recording ----------------
        init_video_writers(obs)
        task.step = recording_step(task.step)                 # wrap step
        task.get_observation = recording_get_observation(     # wrap obs
            task.get_observation
        )

        # ------------------------------------------------
        # 2) Automatically spot any missing predicate(s)
        # ------------------------------------------------
        missing_preds = _detect_missing_predicates()
        if missing_preds:
            print(f"[Analysis] Missing predicate(s) detected: {missing_preds}")
        else:
            print("[Analysis] No missing predicates detected.")

        # ------------------------------------------------
        # 3) Retrieve all known object positions so we can
        #    perform a basic exploration routine.
        # ------------------------------------------------
        positions = get_object_positions()
        if not positions:
            print("[Warning] get_object_positions() returned an empty dict.")

        # ------------------------------------------------
        # 4) Very Simple Exploration:
        #    For every object we (a) move close, (b) attempt
        #    a pick + place, and (c) attempt a pull (to gain
        #    lock information) — all wrapped in try / except
        #    so that failures do not abort the episode.
        # ------------------------------------------------
        for obj_name, obj_pos in positions.items():
            print(f"\n[Exploration] === Handling object: {obj_name} ===")

            # -- Approach / Move
            try:
                print(f"[Exploration] Moving close to {obj_name}")
                _flex_call(move, env, task, target_pos=obj_pos)
            except Exception as e:
                print(f"[Exploration] move() failed for {obj_name}: {e}")

            # -- Pick
            try:
                print(f"[Exploration] Attempting pick on {obj_name}")
                _flex_call(pick, env, task, target_pos=obj_pos)
            except Exception as e:
                print(f"[Exploration] pick() failed for {obj_name}: {e}")

            # -- Immediately place it back (same spot)
            try:
                print(f"[Exploration] Attempting place on {obj_name}")
                _flex_call(place, env, task, target_pos=obj_pos)
            except Exception as e:
                print(f"[Exploration] place() failed for {obj_name}: {e}")

            # -- Pull (useful for drawers / handles)
            try:
                print(f"[Exploration] Attempting pull on {obj_name}")
                # pull frequently does not require a target_pos, just env / task
                _flex_call(pull, env, task)
            except Exception as e:
                # silently ignore pull errors (most objects are not pull-able)
                pass

        print("\n[Exploration] Exploration phase finished.")
        print("===== Task Complete (Simulation continues until shutdown) =====")

    finally:
        # ------------------------------------------------
        # 5) ALWAYS perform a clean shutdown
        # ------------------------------------------------
        shutdown_environment(env)
        print("===== Environment shut down. Script Finished. =====")


if __name__ == "__main__":
    run_skeleton_task()