# run_skeleton_task.py  (completed version – no additional skill definitions)

import time
import inspect
import numpy as np
from pyrep.objects.shape import Shape             # kept – may be useful for future extensions
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import every “primitive” skill that has already been provided externally
from skill_code import *                          # noqa: F403  (we rely ONLY on these!)
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -----------------------------------------------------------
# Helper utilities that do NOT redefine any primitive skills
# -----------------------------------------------------------
def quat_from_euler(roll, pitch, yaw):
    """Convert Euler XYZ to quaternion (x, y, z, w)."""
    cy, sy = np.cos(yaw * 0.5), np.sin(yaw * 0.5)
    cp, sp = np.cos(pitch * 0.5), np.sin(pitch * 0.5)
    cr, sr = np.cos(roll * 0.5), np.sin(roll * 0.5)
    w = cr * cp * cy + sr * sp * sy
    x = sr * cp * cy - cr * sp * sy
    y = cr * sp * cy + sr * cp * sy
    z = cr * cp * sy - sr * sp * cy
    quat = np.array([x, y, z, w], dtype=np.float32)
    return quat / np.linalg.norm(quat)


def call_skill(fn, *args, **kwargs):
    """
    Generic wrapper to call a skill function even when we are unsure
    of its exact signature.  We inspect the parameters and only pass
    the ones the function actually expects.
    """
    sig = inspect.signature(fn)
    accepted = {}
    for k in list(kwargs.keys()):
        if k in sig.parameters:
            accepted[k] = kwargs[k]
    try:
        return fn(*args, **accepted)                      # noqa: F403
    except TypeError as e:
        # If we still have a mismatch, fall back to positional only
        print(f"[WARNING] Skill call signature mismatch for {fn.__name__}: {e}")
        return fn(*args)                                  # noqa: F403


# -----------------------------------------------------------
# MAIN  –  generic skeleton execution
# -----------------------------------------------------------
def run_skeleton_task():
    print("\n===== Starting Skeleton Task =====")
    env, task = setup_environment()                       # initialise simulator
    try:
        descriptions, obs = task.reset()                  # fresh episode
        init_video_writers(obs)                           # optional video writers

        # Wrap step / get_observation so that the video helper can record automatically
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------------------------------------------
        # 1)  Retrieve any known object positions (if provided)
        # ---------------------------------------------------
        positions = {}
        try:
            positions = get_object_positions()            # user-supplied helper
            print(f"[INFO] Retrieved object positions: {list(positions.keys())}")
        except Exception as e:
            print(f"[WARNING] Could not obtain object positions: {e}")

        # ---------------------------------------------------
        # 2)  Exploration Phase  –  identify missing predicate
        # ---------------------------------------------------
        #
        # Feedback told us the missing predicate was  (rotated ?g ?a)
        # so we explicitly perform two rotations (0° and 90°) in order
        # to ground that predicate in the simulator.  We do *not* rely
        # on any predicate checker; instead we just run the skill and
        # assume success if the skill terminates without ‘done = True’.
        #
        print("\n===== Exploration: establishing predicate (rotated) =====")
        exploration_angles = [0.0, np.pi / 2.0]           # 0° and 90° (radians)
        for ang in exploration_angles:
            target_q = quat_from_euler(0.0, 0.0, ang)     # yaw rotation
            print(f"[Exploration] Rotating gripper to yaw={np.degrees(ang):.1f}°")
            obs, reward, done = call_skill(rotate, env, task, target_q)   # noqa: F405
            if done:
                print("[Exploration] Task finished unexpectedly during rotation; aborting.")
                return                                    # safeguard – nothing more to do

        print("[Exploration] Gripper rotations executed.  Predicate ‘rotated’ should now be satisfied.")

        # ---------------------------------------------------
        # 3)  (Optional) Demonstration of a short task plan
        # ---------------------------------------------------
        #
        # If the environment provides drawer-related objects we can
        # execute a minimal open-drawer plan using ONLY the predefined
        # skills.  Everything here is wrapped in ‘try/except’ so the
        # script keeps working even when the expected objects are not
        # present in the current task configuration.
        #
        print("\n===== Attempting minimal drawer-opening plan (if data available) =====")
        try:
            side_pos   = positions.get('drawer_side_pos',   None)
            anchor_pos = positions.get('drawer_anchor_pos', None)
            handle_pos = positions.get('drawer_handle_pos', None)

            if side_pos is not None and anchor_pos is not None and handle_pos is not None:
                # --- move to the side of the drawer (requires rotated ninety_deg) ---
                print("[Plan] Moving to drawer side-position.")
                obs, reward, done = call_skill(move, env, task, target_pos=side_pos)  # noqa: F405
                if done: return

                # --- move straight in to the anchor position ---
                print("[Plan] Moving to drawer anchor-position.")
                obs, reward, done = call_skill(move, env, task, target_pos=anchor_pos)  # noqa: F405
                if done: return

                # --- grip the handle (pick) ---
                print("[Plan] Gripping drawer handle.")
                obs, reward, done = call_skill(pick, env, task, target_pos=handle_pos)  # noqa: F405
                if done: return

                # --- pull the drawer open ---
                print("[Plan] Pulling drawer.")
                obs, reward, done = call_skill(pull, env, task)  # noqa: F405
                if done: return

                print("[SUCCESS] Drawer should now be open!")
            else:
                print("[INFO] Drawer positions were not available – skipping drawer plan.")
        except Exception as e:
            print(f"[WARNING] Drawer-opening plan failed gracefully: {e}")

    finally:
        shutdown_environment(env)                         # always clean up
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
