# run_skeleton_task.py


import os
import re
import time
import glob
import numpy as np

from env import setup_environment, shutdown_environment

# NOTE:
#   – skill_code already provides every primitive (move, pick, place, rotate, pull, …).
#   – we import nothing from it here, because the current task is only to detect the
#     missing predicate that breaks planning.  If, in the future, you need to call any
#     primitive, simply add for example “from skill_code import move”.
#
#   from skill_code import move, pick, place, rotate, pull
#
#   (left commented on purpose – nothing else has to be added here.)


# ---------------------------------------------------------------------------
# Utility ❶  :  simple PDDL-parsing helpers
# ---------------------------------------------------------------------------

_RE_PARENTHESES_TOKEN = re.compile(r'\([^\)]+\)')          # every (…) expression
_RE_ATOM              = re.compile(r'[A-Za-z0-9_\-]+')      # individual tokens


_LOGIC_KEYWORDS = {
    'and', 'or', 'not', 'forall', 'exists', 'when',
    'imply', 'implies', 'either', '=', '>', '<', '<=',
    '>=', 'increase', 'decrease', 'assign', 'total-cost',
    ':condition', ':effect'
}


def _extract_declared_predicates(domain_string: str):
    """
    Return a set with every predicate name that appears in the (:predicates …) block.
    """
    declared = set()

    # Locate the (:predicates …) block (may span multiple lines)
    predicates_block = ''
    paren_level = 0
    reading      = False

    for line in domain_string.splitlines():
        if '(:predicates' in line:
            reading = True

        if reading:
            # update parentheses level so that we know when the block ends
            paren_level += line.count('(')
            paren_level -= line.count(')')
            predicates_block += ' ' + line

            if paren_level == 0:        # finished reading (:predicates …)
                break

    # Every “(pred …)” element in that block
    for expr in _RE_PARENTHESES_TOKEN.findall(predicates_block):
        # The first token after '(' is the predicate name
        tokens = _RE_ATOM.findall(expr)
        if tokens:
            name = tokens[0]
            if name not in _LOGIC_KEYWORDS:
                declared.add(name)

    return declared


def _extract_used_predicates(domain_string: str):
    """
    Returns a set of predicate names that are referenced in all :precondition / :effect
    parts of every action in the domain.
    """
    used = set()
    in_action_block = False
    in_pre_eff      = False
    paren_level     = 0

    for line in domain_string.splitlines():
        # Detect start of an action
        if line.lstrip().startswith('(:action'):
            in_action_block = True
            continue

        if in_action_block and (':precondition' in line or ':effect' in line):
            in_pre_eff  = True

        if in_pre_eff:
            paren_level += line.count('(')
            paren_level -= line.count(')')

            # extract all atoms on this line
            for token in _RE_ATOM.findall(line):
                token_low = token.lower()
                if token_low not in _LOGIC_KEYWORDS:
                    used.add(token_low)

            # when paren_level reaches 0, this :precondition / :effect section ended
            if paren_level == 0:
                in_pre_eff = False

        # action block ends with a closing parenthesis alone in a line
        if in_action_block and line.strip() == ')':
            in_action_block = False

    return used


def find_missing_predicates(domain_string: str):
    """
    Detect predicates that are referenced in actions but never declared in the
    (:predicates …) block.  Return them as a sorted list (unique).
    """
    declared = _extract_declared_predicates(domain_string)
    used     = _extract_used_predicates(domain_string)

    # normalise to lowercase for a fair comparison (PDDL is case-insensitive)
    declared_low = {p.lower() for p in declared}
    missing      = sorted(p for p in used if p not in declared_low)

    return missing


# ---------------------------------------------------------------------------
# Utility ❷  :  locate a .pddl domain file automatically (best effort)
# ---------------------------------------------------------------------------

def _auto_read_domain_file():
    """
    Attempt to find and read a single *.pddl file that contains “(define (domain …”.
    This helper is only a convenience so the same code works inside / outside the
    grading environment.
    """
    # 1) explicit environment variable, if any
    env_path = os.environ.get('DOMAIN_PDDL')
    if env_path and os.path.isfile(env_path):
        with open(env_path, 'r', encoding='utf-8') as fh:
            return fh.read()

    # 2) search in current working directory (and one level below)
    candidates = []
    for pattern in ('*.pddl', '*/*.pddl', '*/*/*.pddl'):
        for path in glob.glob(pattern):
            try:
                with open(path, 'r', encoding='utf-8') as fh:
                    txt = fh.read()
                    if '(define' in txt and '(domain' in txt:
                        candidates.append((path, txt))
            except Exception:
                pass

    # heuristically pick the first “domain*.pddl”, then the first anything
    if candidates:
        domain_first = [c for c in candidates if os.path.basename(c[0]).startswith('domain')]
        chosen = domain_first[0][1] if domain_first else candidates[0][1]
        return chosen

    return ''    # could not locate any domain file


# ---------------------------------------------------------------------------
#  Main entry-point – Skeleton Task
# ---------------------------------------------------------------------------

def run_skeleton_task():
    print("\n===== Starting Skeleton Task =====")

    # ▶ 1. Set-up the simulator as the original skeleton expected
    env, task = setup_environment()

    try:
        descriptions, obs = task.reset()        # RLBench standard reset

        # ----------------------------------------------------------------------------
        # ▶ 2. Exploration phase – identify the missing predicate
        # ----------------------------------------------------------------------------
        print("\n--- Exploration: scanning PDDL domain for missing predicates ---")

        domain_text = _auto_read_domain_file()

        if not domain_text.strip():
            print("WARNING: Could not automatically locate a domain .pddl file. "
                  "Falling back to the exploration domain string shipped in the prompt.")
            # Fallback: use the “exploration” domain supplied in the instructions
            domain_text = '''
            (define (domain exploration)
              (:requirements :strips :typing :conditional-effects :universal-preconditions)
              (:types robot object location)
              (:predicates
                (robot-at ?r - robot ?loc - location)
                (at ?obj - object ?loc - location)
                (identified ?obj - object)
                (temperature-known ?obj - object)
                (holding ?obj - object)
                (handempty)
                (weight-known ?obj - object)
                (durability-known ?obj - object)
              )
              (:action move
                :parameters (?r - robot ?from - location ?to - location)
                :precondition (robot-at ?r ?from)
                :effect (and
                  (not (robot-at ?r ?from))
                  (robot-at ?r ?to)
                  (forall (?obj - object)
                    (when (at ?obj ?to)
                      (identified ?obj)
                    )
                  )
                )
              )
              (:action move
                :parameters (?r - robot ?from - location ?to - location)
                :precondition (robot-at ?r ?from)
                :effect (and
                  (not (robot-at ?r ?from))
                  (robot-at ?r ?to)
                  (forall (?obj - object)
                    (when (at ?obj ?to)
                      (temperature-known ?obj)
                    )
                  )
                )
              )
              (:action pick
                :parameters (?r - robot ?obj - object ?loc - location)
                :precondition (and
                   (robot-at ?r ?loc)
                   (at ?obj ?loc)
                   (handempty)
                )
                :effect (and
                  (holding ?obj)
                  (not (handempty))
                  (not (at ?obj ?loc))
                  (weight-known ?obj)
                )
              )
              (:action pick
                :parameters (?r - robot ?obj - object ?loc - location)
                :precondition (and
                   (robot-at ?r ?loc)
                   (at ?obj ?loc)
                   (handempty)
                )
                :effect (and
                  (holding ?obj)
                  (not (handempty))
                  (not (at ?obj ?loc))
                  (durability-known ?obj)
                )
              )
              (:action pull
                :parameters (?r - robot ?obj - object ?loc - location)
                :precondition (and
                   (robot-at ?r ?loc)
                   (at ?obj ?loc)
                   (holding ?obj)
                   (not (lock-known ?obj))
                )
                :effect (lock-known ?obj)
              )
            )
            '''

        missing_predicates = find_missing_predicates(domain_text)

        if missing_predicates:
            print(f"[Result] Missing predicate(s) detected: {', '.join(missing_predicates)}")
        else:
            print("[Result] No missing predicate found – domain predicates appear consistent.")

        # ----------------------------------------------------------------------------
        # ▶ 3. (Optional) If you later wish to demonstrate skills, insert calls here
        # ----------------------------------------------------------------------------
        #
        # For this particular submission the assignment focuses only on discovering the
        # domain inconsistency, thus no low-level robot skills are executed.  However,
        # the environment has been correctly initialised and shutdown afterwards, so
        # extending this skeleton into a full task controller would be trivial: simply
        # import and call the appropriate skills (pick, place, move, …) as needed.
        #

    finally:
        shutdown_environment(env)     # always tidy-up

    print("===== End of Skeleton Task =====\n")


if __name__ == "__main__":
    run_skeleton_task()