[Frozen Code Start]

[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        # We interpret this as rotating the gripper 90 degrees about the z‐axis
        print("[Task] Step 1: Rotating gripper by 90 degrees about Z")
        target_quat = R.from_euler('z', 90.0, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Ended after rotation. Reward:", reward)
            return

        # Step 2: Move gripper to the bottom drawer side position
        print("[Task] Step 2: Moving to bottom side position")
        side_bottom = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, target_pos=side_bottom)
        if done:
            print("[Task] Ended after move to side position. Reward:", reward)
            return

        # Step 3: Move gripper to the bottom drawer anchor position
        print("[Task] Step 3: Moving to bottom anchor position")
        anchor_bottom = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, target_pos=anchor_bottom)
        if done:
            print("[Task] Ended after move to anchor position. Reward:", reward)
            return

        # Step 4: Pull the drawer open along the dominant axis
        print("[Task] Step 4: Pulling drawer open")
        vector = anchor_bottom - side_bottom
        idx = int(np.argmax(np.abs(vector)))
        axis_name = ['x','y','z'][idx]
        if vector[idx] < 0:
            axis_name = '-' + axis_name
        distance = float(abs(vector[idx]))
        obs, reward, done = pull(env, task, pull_distance=distance, pull_axis=axis_name)
        if done:
            print("[Task] Ended after pull. Reward:", reward)
            return

        # Step 5: Pick up the rubbish on the table
        print("[Task] Step 5: Picking up rubbish from table")
        rubbish_pos = np.array(positions['rubbish'])
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Ended after pick. Reward:", reward)
            return

        # Step 6: Place the rubbish in the bin
        print("[Task] Step 6: Placing rubbish in bin")
        bin_pos = np.array(positions['bin'])
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task did not signal done (done=False). Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()