import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper to ninety degrees around Z axis
        try:
            print("[Task] Step 1/4: Rotating gripper to 90 degrees around Z axis")
            from scipy.spatial.transform import Rotation as R
            target_quat = R.from_euler('z', 90, degrees=True).as_quat()
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Episode ended during rotate.")
                return
        except Exception as e:
            print("[Task] Exception during rotate:", e)
            return

        # Step 2: Pull open the drawer
        try:
            print("[Task] Step 2/4: Pulling the drawer open")
            PULL_DISTANCE = 0.1
            obs, reward, done = pull(env, task, PULL_DISTANCE, pull_axis='x')
            if done:
                print("[Task] Episode ended during pull.")
                return
        except Exception as e:
            print("[Task] Exception during pull:", e)
            return

        # Step 3: Pick up the rubbish (item3)
        try:
            print("[Task] Step 3/4: Picking up the rubbish (item3)")
            rubbish_pos = positions.get('item3')
            if rubbish_pos is None:
                raise KeyError("Position for 'item3' not found")
            obs, reward, done = pick(env, task, rubbish_pos)
            if done:
                print("[Task] Episode ended during pick.")
                return
        except Exception as e:
            print("[Task] Exception during pick:", e)
            return

        # Step 4: Place the rubbish into the bin
        try:
            print("[Task] Step 4/4: Placing the rubbish into the bin")
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise KeyError("Position for 'bin' not found")
            obs, reward, done = place(env, task, bin_pos)
            if done:
                print("[Task] Episode ended during place.")
                return
        except Exception as e:
            print("[Task] Exception during place:", e)
            return

        print("[Task] Completed all plan steps. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
