[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Task Plan Execution ===
        try:
            # Step 1: Rotate gripper from zero_deg to ninety_deg about z-axis
            print("[Task] Step 1: Rotating gripper to ninety_deg")
            # Compute a quaternion representing a 90° rotation around Z
            target_quat = R.from_euler('z', 90, degrees=True).as_quat()
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Terminated after rotate. Reward:", reward)
                return

            # Step 2: Move to side position at bottom drawer
            print("[Task] Step 2: Moving to bottom side position")
            side_pos = positions['bottom_side_pos']
            obs, reward, done = move(env, task, np.array(side_pos))
            if done:
                print("[Task] Terminated after move-to-side. Reward:", reward)
                return

            # Step 3: Move to anchor position at bottom drawer
            print("[Task] Step 3: Moving to bottom anchor position")
            anchor_pos = positions['bottom_anchor_pos']
            obs, reward, done = move(env, task, np.array(anchor_pos))
            if done:
                print("[Task] Terminated after move-to-anchor. Reward:", reward)
                return

            # Step 4: Pick the drawer handle at anchor position
            print("[Task] Step 4: Picking the drawer handle")
            obs, reward, done = pick(env, task, np.array(anchor_pos))
            if done:
                print("[Task] Terminated after pick-drawer. Reward:", reward)
                return

            # Step 5: Pull the drawer open
            print("[Task] Step 5: Pulling the drawer open")
            # Estimate pull distance as distance between anchor and joint positions
            joint_pos = np.array(positions['bottom_joint_pos'])
            pull_distance = np.linalg.norm(joint_pos - np.array(anchor_pos))
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Terminated after pull. Reward:", reward)
                return

            # Step 6: Pick up the rubbish on the table
            print("[Task] Step 6: Picking up the rubbish")
            rubbish_pos = positions['rubbish']
            obs, reward, done = pick(env, task, np.array(rubbish_pos))
            if done:
                print("[Task] Terminated after pick rubbish. Reward:", reward)
                return

            # Step 7: Place the rubbish into the bin
            print("[Task] Step 7: Placing rubbish into bin")
            bin_pos = positions['bin']
            obs, reward, done = place(env, task, np.array(bin_pos))
            if done:
                print("[Task] Completed task successfully! Reward:", reward)
            else:
                print("[Task] Task ended without done=True. Reward:", reward)

        except Exception as e:
            print(f"[Task] Exception during execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()