[Frozen Code Start]

[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Map angle names to numeric degrees
        angle_map = {
            'zero_deg': 0.0,
            'ninety_deg': 90.0
        }

        # === Execute Oracle Plan Steps ===

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        quat_target = R.from_euler('z', angle_map['ninety_deg'], degrees=True).as_quat()
        obs, reward, done = rotate(env, task, quat_target)
        if done:
            print("[Plan] Terminated after rotate. Reward:", reward)
            return

        # Step 2: move-to-side gripper to side-pos-middle
        print("[Plan] Step 2: move to side-pos-middle")
        if 'side_pos_middle' in positions:
            side_pos = positions['side_pos_middle']
        else:
            side_pos = positions.get('side-pos-middle') or positions.get('side-pos-middle'.replace('-', '_'))
        obs, reward, done = move(env, task, np.array(side_pos))
        if done:
            print("[Plan] Terminated after move-to-side. Reward:", reward)
            return

        # Step 3: move-to-anchor gripper to anchor-pos-middle
        print("[Plan] Step 3: move to anchor-pos-middle")
        if 'anchor_pos_middle' in positions:
            anchor_pos = positions['anchor_pos_middle']
        else:
            anchor_pos = positions.get('anchor-pos-middle') or positions.get('anchor-pos-middle'.replace('-', '_'))
        obs, reward, done = move(env, task, np.array(anchor_pos))
        if done:
            print("[Plan] Terminated after move-to-anchor. Reward:", reward)
            return

        # Step 4: pick-drawer (use pick skill at the anchor handle)
        print("[Plan] Step 4: pick drawer handle at anchor-pos-middle")
        obs, reward, done = pick(env, task, np.array(anchor_pos))
        if done:
            print("[Plan] Terminated after pick-drawer. Reward:", reward)
            return

        # Step 5: pull gripper to open drawer
        print("[Plan] Step 5: pull to open drawer")
        # Assume a default pull distance of 0.1m along x-axis
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Terminated after pull. Reward:", reward)
            return

        # Step 6: pick rubbish from table
        print("[Plan] Step 6: pick rubbish from table")
        rubbish_pos = positions.get('rubbish') or positions.get('rubbish'.replace('-', '_'))
        obs, reward, done = pick(env, task, np.array(rubbish_pos))
        if done:
            print("[Plan] Terminated after pick rubbish. Reward:", reward)
            return

        # Step 7: place rubbish into bin
        print("[Plan] Step 7: place rubbish into bin")
        bin_pos = positions.get('bin') or positions.get('bin'.replace('-', '_'))
        obs, reward, done = place(env, task, np.array(bin_pos))
        if done:
            print("[Plan] Completed task successfully! Reward:", reward)
        else:
            print("[Plan] Finished plan steps. Final done flag:", done, "Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()