import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running the combined drawer‐opening and disposal task.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap task.step and task.get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Positions used in the plan:
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # === Define quaternions for zero and ninety degree rotations ===
        # Here we assume rotation about the Z axis for illustration.
        quat_map = {
            'zero_deg':   np.array([0.0, 0.0, 0.0, 1.0]),
            'ninety_deg': np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        }

        # === Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=quat_map['ninety_deg'])
        if done:
            print("[Plan] Terminated after Step 1")
            return

        # Step 2: Move gripper to the side of the bottom drawer
        print("[Plan] Step 2: move to bottom_side_pos")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Plan] Terminated after Step 2")
            return

        # Step 3: Move gripper to the anchor position of the bottom drawer handle
        print("[Plan] Step 3: move to bottom_anchor_pos")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Terminated after Step 3")
            return

        # Step 4: Pick the drawer handle (simulate pick-drawer)
        print("[Plan] Step 4: pick drawer handle at bottom_anchor_pos")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Terminated after Step 4")
            return

        # Step 5: Pull the drawer open
        print("[Plan] Step 5: pull drawer open")
        # Here we choose a pull distance of 0.1m along the default 'x' axis
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Terminated after Step 5")
            return

        # Step 6: Pick up the rubbish on the table
        print("[Plan] Step 6: pick rubbish from table")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Plan] Terminated after Step 6")
            return

        # Step 7: Place the rubbish into the bin
        print("[Plan] Step 7: place rubbish into bin")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Completed successfully after placing rubbish! Reward:", reward)
        else:
            print("[Task] Finished plan execution; done flag is", done, "Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()