from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define target orientation: 90 degrees about Z axis quaternion [x, y, z, w]
        ninety_quat = np.array([0.0, 0.0, 0.70710678, 0.70710678])

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper to ninety_deg
        print("[Task] Step 1: Rotating gripper to 90 degrees orientation")
        try:
            obs, reward, done = rotate(env, task, ninety_quat)
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Task] Terminated after rotate")
            return

        # Step 2: Move to side position of bottom drawer
        print("[Task] Step 2: Moving to bottom_side_pos")
        try:
            side_pos = np.array(positions['bottom_side_pos'])
            obs, reward, done = move(env, task, side_pos)
        except KeyError:
            print("[Error] bottom_side_pos not found in positions")
            return
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return
        if done:
            print("[Task] Terminated after move to side")
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Task] Step 3: Moving to bottom_anchor_pos")
        try:
            anchor_pos = np.array(positions['bottom_anchor_pos'])
            obs, reward, done = move(env, task, anchor_pos)
        except KeyError:
            print("[Error] bottom_anchor_pos not found in positions")
            return
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return
        if done:
            print("[Task] Terminated after move to anchor")
            return

        # Step 4: Pull the drawer open
        print("[Task] Step 4: Pulling drawer open")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Terminated after pull")
            return

        # Step 5: Pick up the trash (assumed to be item3)
        print("[Task] Step 5: Picking up trash (item3)")
        try:
            trash_pos = np.array(positions['item3'])
            obs, reward, done = pick(env, task, trash_pos)
        except KeyError:
            print("[Error] item3 not found in positions")
            return
        except Exception as e:
            print(f"[Error] pick failed: {e}")
            return
        if done:
            print("[Task] Terminated after pick")
            return

        # Step 6: Place the trash into the bin
        print("[Task] Step 6: Placing trash into bin")
        try:
            bin_pos = np.array(positions['bin'])
            obs, reward, done = place(env, task, bin_pos)
        except KeyError:
            print("[Error] bin not found in positions")
            return
        except Exception as e:
            print(f"[Error] place failed: {e}")
            return
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Execution finished (done=False). Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()