from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    '''Generic skeleton for running the drawer‐opening and rubbish disposal task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # Wrap step & get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Key positions for bottom drawer manipulation
        drawer_side_pos   = positions['bottom_side_pos']
        drawer_anchor_pos = positions['bottom_anchor_pos']
        # Rubbish items and bin
        item1_pos = positions['item1']
        bin_pos   = positions['bin']

        # === Execute Oracle Plan Steps ===

        # 1) Rotate gripper (align handle orientation)
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]
        print("[Plan] Step 1: rotate gripper")
        obs, reward, done = rotate(env, task, target_quat=current_quat)
        if done:
            print("[Plan] Ended after rotate. Reward:", reward)
            return

        # 2) Move to side position of bottom drawer
        print("[Plan] Step 2: move to side position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=np.array(drawer_side_pos))
        if done:
            print("[Plan] Ended after move to side. Reward:", reward)
            return

        # 3) Move to anchor (handle) position of bottom drawer
        print("[Plan] Step 3: move to anchor position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=np.array(drawer_anchor_pos))
        if done:
            print("[Plan] Ended after move to anchor. Reward:", reward)
            return

        # 4) Pull to open the drawer
        print("[Plan] Step 4: pull to open bottom drawer")
        # pull_distance tuned to drawer depth
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Ended after pull. Reward:", reward)
            return

        # 5) Pick up the first item (rubbish) from the table
        print("[Plan] Step 5: pick item1 from table")
        obs, reward, done = pick(env, task, target_pos=np.array(item1_pos))
        if done:
            print("[Plan] Ended after pick. Reward:", reward)
            return

        # 6) Place the picked item into the bin
        print("[Plan] Step 6: place item1 into bin")
        obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
        if done:
            print("[Task] Completed successfully! Final reward:", reward)
        else:
            print("[Task] Plan finished but done=False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()