[Frozen Code Start]
    
[Frozen Code End]

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running the combined-domain task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Map the PDDL names (with hyphens) to our underscore keys
        # Drawer side and anchor positions
        bottom_side_pos   = positions.get('bottom_side_pos')
        bottom_anchor_pos = positions.get('bottom_anchor_pos')
        # Table and bin locations
        table_pos = positions.get('table')
        bin_pos   = positions.get('bin')
        # Rubbish object
        rubbish_pos = positions.get('rubbish')

        # === Oracle Plan Execution ===

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to 90 degrees")
        # Define a target quaternion for 90° about the Y axis
        target_quat = R.from_euler('y', 90.0, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task ended after rotate. Reward:", reward)
            return

        # Step 2: move to drawer side position
        print("[Plan] Step 2: move to side-pos-bottom")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Plan] Task ended after move-to-side. Reward:", reward)
            return

        # Step 3: move to drawer anchor position
        print("[Plan] Step 3: move to anchor-pos-bottom")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after move-to-anchor. Reward:", reward)
            return

        # Step 4: pick-drawer (grasp the drawer handle)
        print("[Plan] Step 4: pick-drawer at anchor-pos-bottom")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after pick-drawer. Reward:", reward)
            return

        # Step 5: pull the drawer open
        print("[Plan] Step 5: pull drawer open")
        # Pull along +X by a small distance (e.g., 0.1m)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Task ended after pull. Reward:", reward)
            return

        # Step 6: pick the rubbish from the table
        print("[Plan] Step 6: move to table then pick rubbish")
        # Move above the table
        obs, reward, done = move(env, task, table_pos)
        if done:
            print("[Plan] Task ended after move to table. Reward:", reward)
            return
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Task ended after pick rubbish. Reward:", reward)
            return

        # Step 7: place the rubbish into the bin
        print("[Plan] Step 7: move to bin then place rubbish")
        obs, reward, done = move(env, task, bin_pos)
        if done:
            print("[Plan] Task ended after move to bin. Reward:", reward)
            return
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Finished plan but not in done=True state. Reward:", reward)

    except Exception as e:
        print("[Error] Exception during task execution:", e)
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
