[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Task: Select a drawer and open it fully, then pick up the rubbish and leave it in the trash can.'''
    print("===== Starting Drawer‐and‐Dispose Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state and set up recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        
        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expected keys in positions dict:
        #   'bottom_side_pos', 'bottom_anchor_pos', 'rubbish', 'bin'
        if not all(k in positions for k in ['bottom_side_pos', 'bottom_anchor_pos', 'rubbish', 'bin']):
            raise KeyError("Missing one of required position keys in positions dict.")
        
        # === PLAN STEP 1: rotate gripper zero_deg ninety_deg ===
        # Compute a 90° rotation about the x‐axis
        quat_90_x = R.from_euler('x', 90, degrees=True).as_quat()
        print("[Plan] Step 1: rotate gripper to 90° about x-axis.")
        obs, reward, done = rotate(env, task, quat_90_x)
        if done:
            print("[Plan] Terminated early during rotate. Reward:", reward)
            return

        # === PLAN STEP 2: move-to-side gripper bottom nowhere-pos side-pos-bottom ===
        print("[Plan] Step 2: move to drawer side position.")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Terminated early during move-to-side. Reward:", reward)
            return

        # === PLAN STEP 3: move-to-anchor gripper bottom side-pos-bottom anchor-pos-bottom ===
        print("[Plan] Step 3: move to drawer anchor position.")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Terminated early during move-to-anchor. Reward:", reward)
            return

        # === PLAN STEP 4: pick-drawer gripper bottom anchor-pos-bottom ===
        print("[Plan] Step 4: pick drawer handle.")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Plan] Terminated early during pick-drawer. Reward:", reward)
            return

        # === PLAN STEP 5: pull gripper bottom ===
        print("[Plan] Step 5: pull drawer open.")
        # Pull distance chosen heuristically (adjust if needed)
        pull_distance = 0.15
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Plan] Terminated early during pull. Reward:", reward)
            return

        # === PLAN STEP 6: pick rubbish table ===
        print("[Plan] Step 6: pick up rubbish.")
        rubbish_pos = np.array(positions['rubbish'])
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Terminated early during pick rubbish. Reward:", reward)
            return

        # === PLAN STEP 7: place rubbish bin ===
        print("[Plan] Step 7: place rubbish into bin.")
        bin_pos = np.array(positions['bin'])
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Plan] Completed by placing rubbish. Reward:", reward)
        else:
            print("[Plan] Final step executed. Reward:", reward)

    except Exception as e:
        print("[Error] Exception during task execution:", str(e))
    finally:
        shutdown_environment(env)
        print("===== End of Drawer‐and‐Dispose Task =====")


if __name__ == "__main__":
    run_skeleton_task()