[Frozen Code Start]

[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        
        # Wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        
        # === Retrieve Object Positions and Orientations ===
        positions = get_object_positions()
        # Some keys may use hyphens or underscores; try both forms
        def lookup(key1, key2=None):
            if key1 in positions:
                return np.array(positions[key1])
            if key2 and key2 in positions:
                return np.array(positions[key2])
            raise KeyError(f"Position for {key1} not found.")
        
        bottom_side_pos = lookup('bottom_side_pos', 'side-pos-bottom')
        bottom_anchor_pos = lookup('bottom_anchor_pos', 'anchor-pos-bottom')
        rubbish_pos = lookup('rubbish')
        bin_pos = lookup('bin')
        
        # Define quaternions for orientations
        zero_quat = R.from_euler('z', 0, degrees=True).as_quat()    # [x, y, z, w]
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()
        
        # === Execute Oracle Plan ===
        
        # Step 1: rotate gripper from 0° to 90°
        print("[Task] Step 1/7: Rotating gripper to 90 degrees")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Ended after rotate")
            return
        
        # Step 2: move to side position of bottom drawer
        print(f"[Task] Step 2/7: Moving gripper to side position: {bottom_side_pos}")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Ended after move-to-side")
            return
        
        # Step 3: move to anchor position of bottom drawer
        print(f"[Task] Step 3/7: Moving gripper to anchor position: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended after move-to-anchor")
            return
        
        # Step 4: pick the drawer handle (simulate pick-drawer)
        print(f"[Task] Step 4/7: Grasping drawer handle at: {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.02)
        if done:
            print("[Task] Ended after pick-drawer")
            return
        
        # Step 5: pull the drawer open
        print("[Task] Step 5/7: Pulling drawer open")
        # Choose a reasonable pull distance (e.g., 0.2m along x-axis)
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Ended after pull")
            return
        
        # Step 6: pick up rubbish from table
        print(f"[Task] Step 6/7: Picking up rubbish at: {rubbish_pos}")
        obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15)
        if done:
            print("[Task] Ended after pick rubbish")
            return
        
        # Step 7: place rubbish into the bin
        print(f"[Task] Step 7/7: Placing rubbish into bin at: {bin_pos}")
        obs, reward, done = place(env, task, bin_pos, approach_distance=0.15)
        if done:
            print("[Task] Ended after place rubbish")
            return
        
        print("[Task] Completed all steps. Final reward:", reward)
    
    except Exception as e:
        print("[Task] Exception during execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()