[Frozen Code Start]

[Frozen Code End]

import numpy as np
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from skill_code import move, pick, place, rotate, pull

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset task and initialize recording if needed
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all object and target positions
        positions = get_object_positions()

        try:
            # Step 1: rotate gripper from zero_deg to ninety_deg
            # We generate a quaternion representing a 90° rotation around Z
            target_quat = np.array([0.0, 0.0,
                                    np.sin(np.deg2rad(45.0)),
                                    np.cos(np.deg2rad(45.0))])
            print("[Task] Step 1: rotate to ninety_deg", target_quat)
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Early exit after rotate. Reward:", reward)
                return

            # Step 2: move-to-side (gripper -> side position of bottom drawer)
            side_pos = positions['bottom_side_pos']
            print("[Task] Step 2: move to side-pos-bottom", side_pos)
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Early exit after move-to-side. Reward:", reward)
                return

            # Step 3: move-to-anchor (gripper -> anchor position of bottom drawer)
            anchor_pos = positions['bottom_anchor_pos']
            print("[Task] Step 3: move to anchor-pos-bottom", anchor_pos)
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Early exit after move-to-anchor. Reward:", reward)
                return

            # Step 4: pick-drawer (pick at the anchor position to grasp drawer handle)
            print("[Task] Step 4: pick-drawer at anchor-pos-bottom", anchor_pos)
            obs, reward, done = pick(env, task, anchor_pos)
            if done:
                print("[Task] Early exit after pick-drawer. Reward:", reward)
                return

            # Step 5: pull (open the drawer)
            # We choose a small pull distance along the +X axis
            pull_distance = 0.1
            print(f"[Task] Step 5: pull drawer by {pull_distance} m along +X")
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Early exit after pull. Reward:", reward)
                return

            # Step 6: pick rubbish from table
            rubbish_pos = positions['rubbish']
            print("[Task] Step 6: pick rubbish at", rubbish_pos)
            obs, reward, done = pick(env, task, rubbish_pos)
            if done:
                print("[Task] Early exit after picking rubbish. Reward:", reward)
                return

            # Step 7: place rubbish into bin
            bin_pos = positions['bin']
            print("[Task] Step 7: place rubbish in bin at", bin_pos)
            obs, reward, done = place(env, task, bin_pos)
            if done:
                print("[Task] Completed after place. Reward:", reward)
            else:
                print("[Task] Plan finished. Reward:", reward)

        except Exception as e:
            print("[Task] Exception during execution:", e)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()