[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Task logic implementing the oracle plan.'''
    print("===== Starting Combined Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expected keys in positions: 'bottom_side_pos', 'bottom_anchor_pos', 'rubbish', 'table', 'bin'
        # Some environments name 'table' or 'waypoint1'; adjust if necessary.
        if 'table' not in positions:
            # If table not returned, try looking for any surface object
            raise KeyError("Table position not found in positions dictionary")

        # === Execute Oracle Plan ===
        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to 90 degrees")
        # Compute a quaternion for a 90° rotation around the X-axis (XYZW order)
        target_quat = R.from_euler('x', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Ended during rotate")
            return

        # Step 2: move-to-side (approach bottom drawer side)
        print("[Plan] Step 2: move gripper to side of bottom drawer")
        side_pos = positions['bottom_side_pos']
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Plan] Ended during move-to-side")
            return

        # Step 3: move-to-anchor (move to bottom drawer handle)
        print("[Plan] Step 3: move gripper to anchor position on bottom drawer")
        anchor_pos = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Plan] Ended during move-to-anchor")
            return

        # Step 4: pick-drawer (close gripper around drawer handle)
        print("[Plan] Step 4: grasp the drawer handle")
        # We use the pick primitive on the handle location
        obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Plan] Ended during pick-drawer")
            return

        # Step 5: pull drawer open
        print("[Plan] Step 5: pull the drawer open")
        # We choose a nominal pull distance, e.g., 0.1 meters along the drawer opening axis
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Ended during pull")
            return

        # Step 6: pick up any rubbish on the table
        print("[Plan] Step 6: pick up rubbish from the table")
        rubbish_pos = positions.get('rubbish')
        if rubbish_pos is None:
            raise KeyError("Rubbish position not found")
        obs, reward, done = pick(env, task, target_pos=np.array(rubbish_pos))
        if done:
            print("[Plan] Ended during pick rubbish")
            return

        # Step 7: place rubbish into bin
        print("[Plan] Step 7: place rubbish into bin")
        bin_pos = positions.get('bin')
        if bin_pos is None:
            raise KeyError("Bin position not found")
        obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
        if done:
            print("[Plan] Ended during place rubbish")
        else:
            print(f"[Plan] Completed all steps. Final reward: {reward}")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Combined Task =====")


if __name__ == "__main__":
    run_skeleton_task()