import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

# [Frozen Code Start]

# [Frozen Code End]

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===
        try:
            # Step 1: Rotate gripper from zero_deg to ninety_deg
            print("[Plan] Step 1: Rotate gripper to ninety_deg")
            # Assuming rotation around Y axis by 90 degrees corresponds to ninety_deg
            target_quat = R.from_euler('y', 90, degrees=True).as_quat()
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Plan] Early termination after rotate. Reward:", reward)
                return

            # Step 2: Move to the side position of the bottom drawer
            print("[Plan] Step 2: Move to bottom_side_pos")
            side_pos = np.array(positions['bottom_side_pos'])
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Plan] Early termination after move-to-side. Reward:", reward)
                return

            # Step 3: Move to the anchor position of the bottom drawer
            print("[Plan] Step 3: Move to bottom_anchor_pos")
            anchor_pos = np.array(positions['bottom_anchor_pos'])
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Plan] Early termination after move-to-anchor. Reward:", reward)
                return

            # Step 4: Pick the drawer handle at anchor position
            print("[Plan] Step 4: Pick drawer handle (pick-drawer)")
            obs, reward, done = pick(env, task, anchor_pos)
            if done:
                print("[Plan] Early termination after pick-drawer. Reward:", reward)
                return

            # Step 5: Pull drawer open
            print("[Plan] Step 5: Pull drawer open")
            # pull_distance and axis should match drawer orientation; we assume x-axis and 0.1m
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
            if done:
                print("[Plan] Early termination after pull. Reward:", reward)
                return

            # Step 6: Pick up the trash (rubbish)
            print("[Plan] Step 6: Pick rubbish")
            rubbish_pos = np.array(positions['rubbish'])
            obs, reward, done = pick(env, task, rubbish_pos)
            if done:
                print("[Plan] Early termination after pick rubbish. Reward:", reward)
                return

            # Step 7: Place the trash into the bin
            print("[Plan] Step 7: Place rubbish into bin")
            bin_pos = np.array(positions['bin'])
            obs, reward, done = place(env, task, bin_pos)
            if done:
                print("[Task] Completed successfully! Final reward:", reward)
            else:
                print("[Task] Finished all steps but done flag is False. Final reward:", reward)

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")
            raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()