import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Prepare quaternions for rotation
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        step = 1
        try:
            # Step 1: Rotate gripper from zero_deg to ninety_deg
            print(f"[Step {step}] Rotating gripper to 90 degrees about Z-axis")
            obs, reward, done = rotate(env, task, ninety_quat)
            if done:
                print(f"Early termination after Step {step}")
                return

            # Step 2: Move to side position of the bottom drawer
            step += 1
            side_pos = positions['bottom_side_pos']
            print(f"[Step {step}] Moving to bottom side position: {side_pos}")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print(f"Early termination after Step {step}")
                return

            # Step 3: Move to anchor position of the bottom drawer
            step += 1
            anchor_pos = positions['bottom_anchor_pos']
            print(f"[Step {step}] Moving to bottom anchor position: {anchor_pos}")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print(f"Early termination after Step {step}")
                return

            # Step 4: Pick the drawer handle at the anchor position
            step += 1
            print(f"[Step {step}] Picking drawer handle at: {anchor_pos}")
            obs, reward, done = pick(env, task, anchor_pos)
            if done:
                print(f"Early termination after Step {step}")
                return

            # Step 5: Pull the drawer open by 0.1 m along X axis
            step += 1
            print(f"[Step {step}] Pulling drawer open by 0.1 m along X-axis")
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
            if done:
                print(f"Early termination after Step {step}")
                return

            # Step 6: Pick any remaining rubbish on the table
            step += 1
            rubbish_pos = positions.get('rubbish')
            if rubbish_pos is None:
                raise KeyError("Rubbish position not found in positions")
            print(f"[Step {step}] Picking rubbish at: {rubbish_pos}")
            obs, reward, done = pick(env, task, rubbish_pos)
            if done:
                print(f"Early termination after Step {step}")
                return

            # Step 7: Place the picked rubbish into the bin
            step += 1
            bin_pos = positions['bin']
            print(f"[Step {step}] Placing rubbish into bin at: {bin_pos}")
            obs, reward, done = place(env, task, bin_pos)
            if done:
                print("Task completed successfully! Final reward:", reward)
            else:
                print("Task ended without done=True. Final reward:", reward)

        except Exception as e:
            print("Error during plan execution at step", step, ":", e)

    finally:
        shutdown_environment(env)

if __name__ == "__main__":
    run_skeleton_task()