import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_pos_bottom    = np.array(positions['bottom_side_pos'])
        anchor_pos_bottom  = np.array(positions['bottom_anchor_pos'])
        rubbish_pos        = np.array(positions['item3'])
        bin_pos            = np.array(positions['bin'])

        # === Execute Oracle Plan ===
        print("===== Executing Oracle Plan =====")
        # Step 1: move to bottom side position
        print("[Plan] Step 1: move to bottom side position")
        # Frozen Code Start
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        # Frozen Code End
        if done:
            print("[Plan] Terminated after move step.")
            return

        # Step 2: rotate to ninety_deg
        print("[Plan] Step 2: rotate to ninety_deg")
        quat_z_90 = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, target_quat=quat_z_90)
        if done:
            print("[Plan] Terminated after rotate step.")
            return

        # Step 3: move to bottom anchor position
        print("[Plan] Step 3: move to bottom anchor position")
        obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        if done:
            print("[Plan] Terminated after move-to-anchor step.")
            return

        # Step 4: pull to open drawer
        print("[Plan] Step 4: pull to open drawer")
        pull_distance = 0.1  # meters; adjust if necessary
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Plan] Terminated after pull step.")
            return

        # Step 5: pick up rubbish
        print("[Plan] Step 5: pick up rubbish")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Plan] Terminated after pick step.")
            return

        # Step 6: move to bin
        print("[Plan] Step 6: move to bin")
        obs, reward, done = move(env, task, target_pos=bin_pos)
        if done:
            print("[Plan] Terminated after move-to-bin step.")
            return

        # Step 7: place rubbish in bin
        print("[Plan] Step 7: place rubbish in bin")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done or (reward is not None):
            print(f"[Plan] Completed place step. Final reward: {reward}")
        else:
            print("[Plan] Place step executed; done flag is False.")

    except Exception as e:
        print(f"[Error] Exception during plan execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()