import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the combined-domain task.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        required_keys = ['bottom_side_pos', 'bottom_anchor_pos', 'item3', 'bin']
        for key in required_keys:
            if key not in positions:
                raise KeyError(f"Required object position '{key}' not found in positions dictionary")

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper from zero_deg to ninety_deg")
        from scipy.spatial.transform import Rotation as R
        quat_zero = R.from_euler('z', 0, degrees=True).as_quat()
        quat_ninety = R.from_euler('z', 90, degrees=True).as_quat()

        # Step 2: Prepare side position of bottom drawer
        side_pos = np.array(positions['bottom_side_pos'])
        print(f"[Task] Step 2: Moving to bottom drawer side position at {side_pos}")

        # Frozen Code Start
        obs, reward, done = rotate(env, task, quat_ninety)
        obs, reward, done = move(env, task, side_pos)
        # Frozen Code End

        if done:
            print("[Task] Episode ended during rotation or move to side.")
            return

        # Step 3: Move to anchor position of bottom drawer
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        print(f"[Task] Step 3: Moving to bottom drawer anchor position at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Episode ended after move to anchor.")
            return

        # Step 4: Pick drawer handle at anchor position
        print(f"[Task] Step 4: Picking drawer handle at anchor position {anchor_pos}")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Episode ended during drawer pick.")
            return

        # Step 5: Pull the drawer open
        pull_distance = np.linalg.norm(anchor_pos - side_pos)
        print(f"[Task] Step 5: Pulling the drawer by distance {pull_distance:.4f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Episode ended after pull.")
            return

        # Step 6: Pick up trash (item3)
        trash_pos = np.array(positions['item3'])
        print(f"[Task] Step 6: Picking up trash at {trash_pos}")
        obs, reward, done = pick(env, task, trash_pos)
        if done:
            print("[Task] Episode ended during pick of trash.")
            return

        # Step 7: Place trash into bin
        bin_pos = np.array(positions['bin'])
        print(f"[Task] Step 7: Placing trash into bin at {bin_pos}")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print(f"[Task] Task completed successfully! Final reward: {reward}")
        else:
            print(f"[Task] Task ended without success flag. Reward: {reward}")

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()