import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset task and initialize recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Step 1: Rotate gripper from zero_deg to ninety_deg around Y axis
        # Quaternion [x, y, z, w] for 90° about Y is [0, sin(π/4), 0, cos(π/4)]
        target_quat = np.array([0.0, np.sin(np.pi/4), 0.0, np.cos(np.pi/4)])
        print("[Task] Rotating gripper to 90° around Y-axis:", target_quat)

        # Step 2: Prepare to move to side position of bottom drawer
        if 'bottom_side_pos' not in positions:
            raise KeyError("bottom_side_pos not found in positions")
        side_pos = np.array(positions['bottom_side_pos'])
        print("[Task] Moving to bottom drawer side position at:", side_pos)

        # == FROZEN CODE START ==
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        # == FROZEN CODE END ==

        if done:
            print("[Task] Terminated during rotate or move-to-side step. Reward:", reward)
            return

        # Step 3: Move to the anchor (handle) position of the bottom drawer
        if 'bottom_anchor_pos' not in positions:
            raise KeyError("bottom_anchor_pos not found in positions")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        print("[Task] Moving to bottom drawer anchor position at:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Terminated during move-to-anchor step. Reward:", reward)
            return

        # Step 4: Grasp the drawer handle (pick-drawer)
        print("[Task] Grasping drawer handle at:", anchor_pos)
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Terminated during pick-drawer step. Reward:", reward)
            return

        # Step 5: Pull drawer open
        print("[Task] Pulling drawer open")
        # You may need to adjust pull_distance based on drawer dimensions
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Terminated during pull step. Reward:", reward)
            return
        print("[Task] Drawer is now open.")

        # Steps 6 & 7: Pick up rubbish items on the table and place them into the bin
        if 'bin' not in positions:
            raise KeyError("bin not found in positions")
        bin_pos = np.array(positions['bin'])
        rubbish_keys = [k for k in positions.keys() if k.startswith('item')]
        if not rubbish_keys:
            print("[Task] No rubbish items found on the table.")
        for rubbish in rubbish_keys:
            rubbish_pos = np.array(positions[rubbish])
            print(f"[Task] Picking up rubbish '{rubbish}' at:", rubbish_pos)
            obs, reward, done = pick(env, task, rubbish_pos)
            if done:
                print(f"[Task] Terminated during pick of {rubbish}. Reward:", reward)
                return
            print(f"[Task] Placing rubbish '{rubbish}' into bin at:", bin_pos)
            obs, reward, done = place(env, task, bin_pos)
            if done:
                print(f"[Task] Terminated during place of {rubbish}. Reward:", reward)
                return

        print("[Task] All rubbish chucked away. Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
        raise
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()