import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset and prepare recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_bottom_pos = np.array(positions['bottom_side_pos'])
        anchor_bottom_pos = np.array(positions['bottom_anchor_pos'])
        rubbish_pos = np.array(positions['rubbish'])
        bin_pos = np.array(positions['bin'])

        # === Retrieve target orientation quaternion for ninety_deg ===
        ninety_shape = Shape('ninety_deg')
        quat_ninety = np.array(ninety_shape.get_quaternion())

        done = False

        # === Execute Oracle Plan ===
        print("[Step 1] Rotate gripper to ninety_deg")
        try:
            obs, reward, done = rotate(env, task, quat_ninety)
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            raise
        if done:
            print("[Task] ended after rotate.")
            return

        print("[Step 2] Move to side-pos-bottom of bottom drawer")
        try:
            obs, reward, done = move(env, task, side_bottom_pos)
        except Exception as e:
            print(f"[Error] move-to-side failed: {e}")
            raise
        if done:
            print("[Task] ended after move-to-side.")
            return

        print("[Step 3] Move to anchor-pos-bottom of bottom drawer")
        try:
            obs, reward, done = move(env, task, anchor_bottom_pos)
        except Exception as e:
            print(f"[Error] move-to-anchor failed: {e}")
            raise
        if done:
            print("[Task] ended after move-to-anchor.")
            return

        print("[Step 4] Pick drawer handle at anchor-pos-bottom")
        try:
            obs, reward, done = pick(env, task, anchor_bottom_pos)
        except Exception as e:
            print(f"[Error] pick-drawer failed: {e}")
            raise
        if done:
            print("[Task] ended after pick-drawer.")
            return

        print("[Step 5] Pull drawer open")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            raise
        if done:
            print("[Task] ended after pull.")
            return

        print("[Step 6] Pick rubbish from table")
        try:
            obs, reward, done = pick(env, task, rubbish_pos)
        except Exception as e:
            print(f"[Error] pick rubbish failed: {e}")
            raise
        if done:
            print("[Task] ended after picking rubbish.")
            return

        print("[Step 7] Place rubbish into bin")
        try:
            obs, reward, done = place(env, task, bin_pos)
        except Exception as e:
            print(f"[Error] place rubbish failed: {e}")
            raise
        if done:
            print("[Task] completed successfully! Reward:", reward)
        else:
            print("[Task] done flag is False; final reward:", reward)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()