import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper by 90° about X axis")
        target_quat = R.from_euler('x', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Terminated early after rotate.")
            return

        # Step 2: Move to the "side" position of the bottom drawer
        side_bottom = positions['side-pos-bottom']
        print(f"[Plan] Step 2: move to bottom side position at {side_bottom}")
        obs, reward, done = move(env, task, target_pos=side_bottom)
        if done:
            print("[Plan] Terminated early after move-to-side.")
            return

        # Step 3: Move to the "anchor" position of the bottom drawer
        anchor_bottom = positions['anchor-pos-bottom']
        print(f"[Plan] Step 3: move to bottom anchor position at {anchor_bottom}")
        obs, reward, done = move(env, task, target_pos=anchor_bottom)
        if done:
            print("[Plan] Terminated early after move-to-anchor.")
            return

        # Step 4: Pick the drawer handle (akin to pick-drawer)
        print("[Plan] Step 4: pick the drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_bottom)
        if done:
            print("[Plan] Terminated early after pick-drawer.")
            return

        # Step 5: Pull the drawer open along the X axis
        pull_distance = 0.2  # adjust if needed
        pull_axis = 'x'
        print(f"[Plan] Step 5: pull drawer by {pull_distance}m along {pull_axis}-axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Plan] Terminated early after pull.")
            return

        # Step 6: Pick up the trash object
        rubbish_pos = positions['rubbish']
        print(f"[Plan] Step 6: pick up rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Plan] Terminated early after pick rubbish.")
            return

        # Step 7: Place the trash into the bin
        bin_pos = positions['bin']
        print(f"[Plan] Step 7: place rubbish into bin at {bin_pos}")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Finished all steps. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()