import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to initial state and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: Rotating gripper from zero_deg to ninety_deg")
        if 'ninety_deg' in positions:
            target_quat = np.array(positions['ninety_deg'])
        else:
            obs = task.get_observation()
            target_quat = normalize_quaternion(obs.gripper_pose[3:7])
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Early termination after rotate.")
            return

        # Step 2: Move to drawer side position (bottom_side_pos)
        side_key = 'bottom_side_pos'
        if side_key not in positions:
            raise KeyError(f"Position '{side_key}' not found in object positions.")
        side_pos = np.array(positions[side_key])
        print(f"[Plan] Step 2: Moving gripper to drawer side position at: {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Early termination after move-to-side.")
            return

        # Step 3: Move to drawer anchor position (bottom_anchor_pos)
        anchor_key = 'bottom_anchor_pos'
        if anchor_key not in positions:
            raise KeyError(f"Position '{anchor_key}' not found in object positions.")
        anchor_pos = np.array(positions[anchor_key])
        print(f"[Plan] Step 3: Moving gripper to drawer anchor position at: {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Early termination after move-to-anchor.")
            return

        # Step 4: Grasp drawer handle at anchor position
        print(f"[Plan] Step 4: Grasping drawer handle at anchor position: {anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Early termination after pick-drawer.")
            return

        # Step 5: Pull the drawer open along +x axis
        pull_distance = 0.1
        print(f"[Plan] Step 5: Pulling drawer by {pull_distance}m along +x axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Plan] Early termination after pull.")
            return

        # Step 6: Pick up the rubbish (item3)
        rubbish_key = 'item3'
        if rubbish_key not in positions:
            raise KeyError(f"Position '{rubbish_key}' not found in object positions.")
        rubbish_pos = np.array(positions[rubbish_key])
        print(f"[Plan] Step 6: Picking up rubbish at: {rubbish_pos}")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Plan] Early termination after picking rubbish.")
            return

        # Step 7: Place the rubbish in the trash bin
        bin_key = 'bin'
        if bin_key not in positions:
            raise KeyError(f"Position '{bin_key}' not found in object positions.")
        bin_pos = np.array(positions[bin_key])
        print(f"[Plan] Step 7: Placing rubbish into bin at: {bin_pos}")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Task finished (done=False). Final reward:", reward)

    except Exception as e:
        print(f"[Error] {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()