import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        positions = get_object_positions()

        # Step 1: rotate gripper to ninety_deg
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # Step 2: move to bottom side position
        print("[Plan] Step 2: move to bottom side position")
        if 'bottom_side_pos' not in positions:
            raise KeyError("Expected 'bottom_side_pos' in positions")
        side_pos = np.array(positions['bottom_side_pos'])

        [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        [Frozen Code End]

        # Step 3: move to bottom anchor position
        print("[Plan] Step 3: move to bottom anchor position")
        if 'bottom_anchor_pos' not in positions:
            raise KeyError("Expected 'bottom_anchor_pos' in positions")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after move-to-anchor. Reward:", reward)
            return

        # Step 4: pick drawer handle at anchor
        print("[Plan] Step 4: pick drawer handle at anchor")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after pick-drawer. Reward:", reward)
            return

        # Step 5: pull drawer open
        print("[Plan] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Plan] Task ended prematurely after pull. Reward:", reward)
            return

        # Step 6: pick up trash (rubbish)
        print("[Plan] Step 6: pick up trash (rubbish)")
        if 'rubbish' not in positions:
            raise KeyError("Expected 'rubbish' in positions")
        trash_pos = np.array(positions['rubbish'])
        obs, reward, done = pick(env, task, trash_pos)
        if done:
            print("[Plan] Task ended prematurely after pick. Reward:", reward)
            return

        # Step 7: place trash into bin
        print("[Plan] Step 7: place trash into bin")
        if 'bin' not in positions:
            raise KeyError("Expected 'bin' in positions")
        bin_pos = np.array(positions['bin'])
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Plan] Task completed after place. Reward:", reward)
            return

        print("[Plan] Oracle plan executed. Final reward:", reward, "Done flag:", done)

    except Exception as e:
        print("[Error] Exception during task execution:", str(e))
        raise
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()