import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # Precompute quaternions for rotation
        zero_quat   = R.from_euler('x', 0,  degrees=True).as_quat()
        ninety_quat = R.from_euler('x', 90, degrees=True).as_quat()

        # === Oracle Plan Execution ===
        # Step 1: rotate
        obs, reward, done = rotate(env, task, ninety_quat)
        # Step 2: move-to-side
        obs, reward, done = move(env, task, bottom_side_pos)
        # Step 3: move-to-anchor
        obs, reward, done = move(env, task, bottom_anchor_pos)
        # Step 4: pick-drawer (simulated by pick at anchor)
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        # Step 5: pull
        obs, reward, done = pull(env, task, pull_distance=0.05, pull_axis='x')
        # Step 6: pick rubbish
        obs, reward, done = pick(env, task, rubbish_pos)
        # Step 7: place rubbish into bin
        obs, reward, done = place(env, task, bin_pos)

        if done:
            print("[Task] Task succeeded with reward:", reward)
        else:
            print("[Task] Task not completed (done=False). Reward:", reward)

    except Exception as e:
        print(f"[Error] An exception occurred: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()