import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing the simulation
        init_video_writers(obs)

        # Wrap the task.step and task.get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_joint_pos  = positions['bottom_joint_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper by 90 degrees")
        # quaternion for 90° rotation about the x axis: [sin(45°), 0, 0, cos(45°)]
        q_ninety = np.array([0.7071068, 0.0, 0.0, 0.7071068])
        obs, reward, done = rotate(env, task, q_ninety)
        if done:
            print("[Plan] Task ended prematurely after rotate. Reward:", reward)
            return

        # Step 2: Move to side position of bottom drawer
        print(f"[Plan] Step 2: move to side of bottom drawer at {bottom_side_pos}")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Plan] Task ended prematurely after move-to-side. Reward:", reward)
            return

        # Step 3: Move to anchor position of bottom drawer
        print(f"[Plan] Step 3: move to anchor of bottom drawer at {bottom_anchor_pos}")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick the drawer handle
        print(f"[Plan] Step 4: pick drawer handle at {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after pick-drawer. Reward:", reward)
            return

        # Step 5: Pull the drawer open
        pull_distance = np.linalg.norm(bottom_joint_pos - bottom_anchor_pos)
        print(f"[Plan] Step 5: pull drawer open by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task ended prematurely after pull. Reward:", reward)
            return

        # Step 6: Pick up the rubbish from the table
        print(f"[Plan] Step 6: pick rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Task ended prematurely after pick rubbish. Reward:", reward)
            return

        # Step 7: Place the rubbish into the bin
        print(f"[Plan] Step 7: place rubbish into bin at {bin_pos}")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
            return

        # Final check
        print("[Plan] Finished all steps. Final reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
