import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset task and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve necessary object positions and orientations
        positions = get_object_positions()
        zero_quat          = positions['zero_deg']
        ninety_quat        = positions['ninety_deg']
        bottom_side_pos    = positions['bottom_side_pos']
        bottom_anchor_pos  = positions['bottom_anchor_pos']
        rubbish_pos        = positions['rubbish']
        bin_pos            = positions['bin']

        # Step 1: Rotate gripper from zero to ninety degrees
        print("[Task] Step 1: Rotating gripper to 90 degrees")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Task ended after rotation.")
            return

        # Step 2: Move gripper to side position of bottom drawer
        print("[Task] Step 2: Moving gripper to bottom side position")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move-to-side.")
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        print("[Task] Step 3: Moving gripper to bottom anchor position")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move-to-anchor.")
            return

        # Step 4: Pick the drawer handle at the anchor position
        print("[Task] Step 4: Picking drawer handle")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended during pick-drawer.")
            return

        # Step 5: Pull drawer open
        print("[Task] Step 5: Pulling drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Task ended during pull.")
            return

        # Step 6: Pick up the rubbish from the table
        print("[Task] Step 6: Picking up rubbish from table")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Task ended during pick of rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        print("[Task] Step 7: Placing rubbish into bin")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task ended during place of rubbish.")
            return

        # Final status
        if reward is not None:
            print(f"[Task] Completed plan execution. Final reward: {reward}")
        else:
            print("[Task] Plan execution finished (done=False).")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()