import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, place, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to 90 degrees")
        zero_quat   = R.from_euler('xyz', [0, 0,   0], degrees=True).as_quat()
        ninety_quat = R.from_euler('xyz', [0, 0,  90], degrees=True).as_quat()
        [Frozen Code Start]
    obs, reward, done = rotate(env, task, target_quat=ninety_quat)
obs, reward, done = move(env, task, bottom_side_pos)
[Transient whitespace must be kept exactly]
        [Frozen Code End]

        if done:
            print("[Plan] Early termination after rotate.")
            return

        print(f"[Plan] Step 2: move to side position of bottom drawer at {bottom_side_pos}")
        if done:
            print("[Plan] Early termination after move-to-side.")
            return

        # Step 3: Move to anchor position of bottom drawer
        print(f"[Plan] Step 3: move to anchor position of bottom drawer at {bottom_anchor_pos}")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Early termination after move-to-anchor.")
            return

        # Step 4: Pick the drawer handle
        print(f"[Plan] Step 4: pick drawer handle at {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Early termination after pick-drawer.")
            return

        # Step 5: Pull drawer open
        print("[Plan] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Early termination after pull.")
            return

        # Step 6: Pick up the rubbish from the table
        print(f"[Plan] Step 6: pick rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Early termination after pick rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        print(f"[Plan] Step 7: place rubbish into bin at {bin_pos}")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print(f"[Task] Completed successfully! Final reward: {reward}")
        else:
            print(f"[Task] Plan executed but task not marked done. Current reward: {reward}")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()