import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve known object positions
        positions = get_object_positions()
        # Expect keys like 'robot_base', 'drawer_side', 'drawer_anchor', etc.
        if 'robot_base' not in positions or 'drawer_side' not in positions or 'drawer_anchor' not in positions:
            print("Error: Expected keys 'robot_base', 'drawer_side', 'drawer_anchor' in positions.")
            return

        robot_base_pos = positions['robot_base']
        side_pos = positions['drawer_side']
        anchor_pos = positions['drawer_anchor']

        # 1) Rotate gripper from zero to ninety degrees so we can approach the side of the drawer
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Finished early during rotation")
            return

        # 2) Move robot base to a position at the side of the drawer
        print(f"[Task] Moving robot from base at {robot_base_pos} to drawer side at {side_pos}")
        obs, reward, done = move(env, task, from_location=robot_base_pos, to_location=side_pos)
        if done:
            print("[Task] Finished early during move to side")
            return

        # 3) Move robot closer to the anchor on the drawer handle
        print(f"[Task] Moving robot from side at {side_pos} to anchor at {anchor_pos}")
        obs, reward, done = move(env, task, from_location=side_pos, to_location=anchor_pos)
        if done:
            print("[Task] Finished early during move to anchor")
            return

        # 4) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Finished early during pull")
            return

        # 5) Rotate gripper back to zero degrees for any downstream manipulation
        zero_deg_quat = [0.0, 0.0, 0.0, 1.0]
        print("[Task] Rotating gripper back to zero degrees")
        obs, reward, done = rotate(env, task, target_quat=zero_deg_quat)
        if done:
            print("[Task] Finished early during final rotation")
            return

        print("[Task] Drawer successfully opened and gripper reset")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()