import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", list(positions.keys()))

        # Ensure 'bin' is known
        if 'bin' not in positions:
            print("Warning: 'bin' not found in object positions; proceeding without it.")

        # Identify drawer and its side/anchor positions
        drawer_keys = [k for k in positions if 'drawer' in k]
        if not drawer_keys:
            print("Error: No drawer object found in positions. Aborting task.")
            return
        drawer = drawer_keys[0]

        side_pos_keys = [k for k,v in positions.items() if 'side' in k and drawer in k]
        anchor_pos_keys = [k for k,v in positions.items() if 'anchor' in k and drawer in k]
        if not side_pos_keys or not anchor_pos_keys:
            print("Error: Cannot locate side or anchor positions for drawer. Aborting task.")
            return
        side_pos = positions[side_pos_keys[0]]
        anchor_pos = positions[anchor_pos_keys[0]]

        # === Exploration Phase: Determine missing predicate (lock-known) by a test pull ===
        print("[Exploration] Moving robot near drawer to inspect it")
        try:
            # move signature assumed: move(env, task, from_pos, to_pos)
            current_obs = task.get_observation()
            robot_pos = tuple(current_obs.gripper_pose[:3])
            obs, reward, done = move(env, task, robot_pos, side_pos)
            if done:
                print("[Exploration] Early termination after move")
                return
        except Exception as e:
            print(f"[Exploration] Error during move: {e}")
            return

        # attempt to pick drawer handle and pull to learn lock state
        print("[Exploration] Attempting to pick drawer handle for durability/weight inspection")
        try:
            obs, reward, done = pick(env, task, drawer, side_pos)
            if done:
                print("[Exploration] Pick ended the task prematurely.")
                return
        except Exception as e:
            print(f"[Exploration] Error during pick: {e}")
            return

        print("[Exploration] Attempting pull to identify lock state (lock-known predicate)")
        try:
            obs, reward, done = pull(env, task, drawer)
            # we inspect obs for any flag or metadata indicating lock-known
            if hasattr(obs, 'metadata') and 'lock_known' in obs.metadata:
                print("[Exploration] Learned lock-known state:", obs.metadata['lock_known'])
            else:
                print("[Exploration] Pull executed; check environment for lock-known predicate.")
            if done:
                print("[Exploration] Task ended after pull.")
                return
        except Exception as e:
            print(f"[Exploration] Error during pull: {e}")
            return

        # === Oracle Plan Execution: open the drawer ===
        print("[Plan] 1) Rotate gripper to ninety degrees")
        try:
            ninety_deg = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
            obs, reward, done = rotate(env, task, ninety_deg)
            if done:
                print("[Plan] Task ended during rotate")
                return
        except Exception as e:
            print(f"[Plan] Error during rotate: {e}")
            return

        print("[Plan] 2) Move to side position of drawer")
        try:
            current_obs = task.get_observation()
            robot_pos = tuple(current_obs.gripper_pose[:3])
            obs, reward, done = move(env, task, robot_pos, side_pos)
            if done:
                print("[Plan] Task ended after moving to side")
                return
        except Exception as e:
            print(f"[Plan] Error during move-to-side: {e}")
            return

        print("[Plan] 3) Move to anchor position of drawer")
        try:
            obs, reward, done = move(env, task, side_pos, anchor_pos)
            if done:
                print("[Plan] Task ended after moving to anchor")
                return
        except Exception as e:
            print(f"[Plan] Error during move-to-anchor: {e}")
            return

        print("[Plan] 4) Pick the drawer handle")
        try:
            obs, reward, done = pick(env, task, drawer, anchor_pos)
            if done:
                print("[Plan] Task ended after pick-drawer")
                return
        except Exception as e:
            print(f"[Plan] Error during pick-drawer: {e}")
            return

        print("[Plan] 5) Pull the drawer open")
        try:
            obs, reward, done = pull(env, task, drawer)
            if done:
                print("[Plan] Drawer successfully opened; task complete.")
            else:
                print("[Plan] Drawer opened; continuing.")
        except Exception as e:
            print(f"[Plan] Error during pull: {e}")
            return

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()