import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step/get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # assume keys 'object_1', 'object_2', 'bin' exist in the dictionary
        object_1_pos = positions['object_1']
        object_2_pos = positions['object_2']
        bin_pos      = positions['bin']

        # === Oracle Plan Execution ===

        # 1) Pick object_1
        print(f"[Task] Picking object_1 at: {object_1_pos}")
        obs, reward, done = pick(
            env, task,
            target_pos=object_1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after picking object_1.")
            return

        # 2) Place object_1 into bin
        print(f"[Task] Placing object_1 into bin at: {bin_pos}")
        obs, reward, done = place(
            env, task,
            target_pos=bin_pos,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after placing object_1.")
            return

        # 3) Rotate gripper to standard orientation for next pick
        print("[Task] Rotating gripper to target orientation")
        # here we choose identity quaternion (w=1) as an example
        target_quat = [0.0, 0.0, 0.0, 1.0]
        obs, reward, done = rotate(
            env, task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during rotation.")
            return

        # 4) Pick object_2
        print(f"[Task] Picking object_2 at: {object_2_pos}")
        obs, reward, done = pick(
            env, task,
            target_pos=object_2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after picking object_2.")
            return

        # 5) Place object_2 into bin
        print(f"[Task] Placing object_2 into bin at: {bin_pos}")
        obs, reward, done = place(
            env, task,
            target_pos=bin_pos,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after placing object_2.")
            return

        # 6) As an example of pulling, suppose there's a drawer handle at 'handle_pos'
        if 'handle_pos' in positions:
            handle_pos = positions['handle_pos']
            print(f"[Task] Pulling drawer handle at: {handle_pos}")
            obs, reward, done = pull(
                env, task,
                target_pos=handle_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pull.")
                return

        print("[Task] Oracle plan completed successfully.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()