import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import rotate, pick, place, move, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import math

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions in the scene
        positions = get_object_positions()
        # Expect keys like 'drawer_side_pos', 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            print("Error: missing expected drawer position keys in object_positions.")
            return

        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # 1) Inspect initial gripper orientation
        obs = task.get_observation()
        init_quat = obs.gripper_pose[3:7]
        print(f"[Task] Initial gripper quaternion (xyzw): {init_quat}")

        # 2) Ensure gripper is at zero_deg orientation
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0])
        print("[Task] Rotating gripper to zero_deg")
        obs, reward, done = rotate(env, task, zero_quat)
        if done:
            print("[Task] Task ended during zero_deg rotation.")
            return

        # 3) Rotate gripper to ninety_deg orientation so it can approach drawer side
        # quaternion for ninety_deg about z axis
        sin45 = math.sin(math.pi/4)
        cos45 = math.cos(math.pi/4)
        ninety_quat = np.array([0.0, 0.0, sin45, cos45])
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Task ended during ninety_deg rotation.")
            return

        # 4) Exploration: attempt to pull drawer to learn its lock state
        print("[Task] Attempting to pull the drawer to detect lock state")
        try:
            obs, reward, done = pull(env, task)
            print("[Task] pull() succeeded, drawer is unlocked and now open.")
        except Exception as e:
            print(f"[Task] pull() failed, drawer may be locked: {e}")

        # 5) (Optional) If drawer opened, pick any object inside
        if not done:
            if 'inside_object' in positions:
                obj_pos = positions['inside_object']
                print(f"[Task] Picking up object inside drawer at {obj_pos}")
                obs, reward, done = pick(env, task, target_pos=obj_pos,
                                         approach_distance=0.15,
                                         max_steps=100, threshold=0.01,
                                         approach_axis='z', timeout=10.0)
                if done:
                    print("[Task] Task ended during pick of inside object.")
                    return
                print("[Task] Placing object to drop-off location")
                # assume drop-off location is provided
                if 'dropoff_pos' in positions:
                    drop_pos = positions['dropoff_pos']
                    obs, reward, done = place(env, task, target_pos=drop_pos,
                                               approach_distance=0.15,
                                               max_steps=100, threshold=0.01,
                                               approach_axis='z', timeout=10.0)
                    if done:
                        print("[Task] Task ended during place action.")
                        return
                else:
                    print("Warning: no dropoff_pos provided in positions.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()