import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video recording
        init_video_writers(obs)

        # Wrap step and observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # These keys must match those returned by get_object_positions()
        # for the drawer’s side approach and its anchor handle
        drawer_side_pos = np.array(positions['drawer_side_pos'])
        drawer_anchor_pos = np.array(positions['drawer_anchor_pos'])

        # === Exploration Phase ===
        # In order to discover which property is missing (e.g., rotated predicate),
        # we rotate the gripper through a known angle transition once and observe.
        print("[Exploration] Rotating gripper from zero to ninety degrees to identify rotated predicate")
        obs, reward, done = rotate(env, task, from_angle='zero_deg', to_angle='ninety_deg')
        if done:
            print("[Exploration] Task ended prematurely during rotation exploration.")
            return

        # === Main Task Plan: Open the Drawer ===
        # 1) Move to the side of the drawer
        print(f"[Task] Moving to drawer side position at {drawer_side_pos}")
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("[Task] Task ended after moving to side.")
            return

        # 2) Move to the drawer’s anchor position (handle)
        print(f"[Task] Moving to drawer anchor position at {drawer_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Task ended after moving to anchor.")
            return

        # 3) Grasp the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Task ended during grasp.")
            return

        # 4) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull.")
            return

        print("===== Task completed: Drawer is open! =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

if __name__ == "__main__":
    run_skeleton_task()