# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if you want to record
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # TODO: Replace the following placeholders with the actual object keys from your environment
        # For example:
        # drawer_handle_pos = positions['drawer_handle']
        # tomato_pos = positions['tomato']
        # disposal_bin_pos = positions['disposal_bin']

        # === Define Oracle Plan ===
        # This is a placeholder plan. Replace object keys and parameters to match your specific task.
        oracle_plan = [
            # Example: open a drawer
            ("rotate", {
                "target_quat": [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)],
                "max_steps": 100,
                "threshold": 0.01,
                "timeout": 10.0
            }),
            # move close to drawer side position
            ("move", {
                "target_pos": positions.get("drawer_side_pos"),
                "max_steps": 100,
                "threshold": 0.01,
                "timeout": 10.0
            }),
            # move from side to anchor position
            ("move", {
                "target_pos": positions.get("drawer_anchor_pos"),
                "max_steps": 100,
                "threshold": 0.01,
                "timeout": 10.0
            }),
            # pull the drawer open
            ("pull", {
                # some skills take only env, task but if pull needs args, fill them here
            }),
            # pick the object inside the drawer
            ("pick", {
                "target_pos": positions.get("tomato"),
                "approach_distance": 0.15,
                "max_steps": 100,
                "threshold": 0.01,
                "approach_axis": "z",
                "timeout": 10.0
            }),
            # place the object into the disposal bin
            ("place", {
                "target_pos": positions.get("disposal_bin"),
                "approach_distance": 0.15,
                "max_steps": 100,
                "threshold": 0.01,
                "approach_axis": "z",
                "timeout": 10.0
            })
        ]

        # === Execute Oracle Plan ===
        for step_idx, (skill_name, params) in enumerate(oracle_plan, start=1):
            print(f"[Plan] Step {step_idx}: Executing '{skill_name}' with params {params}")
            if skill_name not in globals():
                print(f"Error: Skill '{skill_name}' is not available.")
                break

            skill_fn = globals()[skill_name]
            try:
                # Try calling with keyword args
                obs, reward, done = skill_fn(env, task, **params)
            except TypeError:
                # Fallback to positional arguments if necessary
                obs, reward, done = skill_fn(env, task, *params.values())

            if done:
                print(f"[run] Task ended early at step {step_idx}.")
                break

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()