import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expecting keys 'item1', 'item2', 'item3', and 'bin' in positions

        # List of item names to process
        item_names = ['item1', 'item2', 'item3']
        disposal_bin_name = 'bin'

        # === Oracle Plan Execution ===
        for item_name in item_names:
            if item_name not in positions:
                print(f"[Error] No position found for '{item_name}'. Skipping.")
                continue

            item_pos = positions[item_name]
            print(f"[Task] Picking up {item_name} at position {item_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=item_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] pick() failed for {item_name}: {e}")
                continue

            if done:
                print("[Task] Terminated during pick phase.")
                return

            if disposal_bin_name not in positions:
                print(f"[Error] No position found for disposal bin '{disposal_bin_name}'.")
                break

            bin_pos = positions[disposal_bin_name]
            print(f"[Task] Placing {item_name} into disposal bin at {bin_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=bin_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] place() failed for {item_name}: {e}")
                continue

            if done:
                print("[Task] Terminated during place phase.")
                return

        print("[Task] All items processed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
