import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset to initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step/get_observation for video recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get known object / handle positions
        positions = get_object_positions()
        # assume keys 'drawer' and 'drawer_handle' are defined by object_positions
        if 'drawer_handle' not in positions or 'drawer' not in positions:
            raise KeyError("Expected keys 'drawer' and 'drawer_handle' in positions")

        # === Exploration Phase: check gripper orientation predicate 'rotated g zero_deg' ===
        obs = task.get_observation()
        raw_quat = obs.gripper_pose[3:7]
        # normalize sensor quaternion if helper available
        try:
            from skill_code import normalize_quaternion
            current_quat = normalize_quaternion(raw_quat)
        except ImportError:
            current_quat = raw_quat / np.linalg.norm(raw_quat)

        # build identity quaternion for zero_deg
        zero_deg_quat = np.array([0.0, 0.0, 0.0, 1.0])
        # compute angle difference
        dot = np.dot(current_quat, zero_deg_quat)
        dot = np.clip(dot, -1.0, 1.0)
        angle_diff = 2 * np.arccos(abs(dot))
        print(f"[Exploration] Angle to zero_deg = {angle_diff:.3f} rad")

        # if gripper is not at zero_deg, rotate into that orientation
        if angle_diff > 0.05:
            print("[Exploration] Rotating gripper to zero_deg")
            obs, reward, done = rotate(env, task, zero_deg_quat)
            if done:
                print("[Task] Task ended prematurely after rotate to zero_deg.")
                return

        # === Oracle Plan Phase: open the drawer ===
        # 1) Move close to the drawer handle
        handle_pos = positions['drawer_handle']
        print(f"[Task] Moving near drawer handle at {handle_pos}")
        obs, reward, done = move(env, task, target_pos=handle_pos)
        if done:
            print("[Task] Task ended after move to handle!")
            return

        # 2) Pick the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task,
                                 target_object='drawer_handle',
                                 target_pos=handle_pos,
                                 gripper='gripper')
        if done:
            print("[Task] Task ended after pick handle!")
            return

        # 3) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended after pull!")
            return

        print("[Task] Drawer should now be open.")
    except Exception as e:
        print(f"Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()