import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the drawer‐opening task.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # reset the scene
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        
        # get 3D positions and known quaternions from the scene
        positions = get_object_positions()
        # assume these keys are provided by get_object_positions
        # e.g. 'drawer_handle', 'zero_deg', 'ninety_deg'
        if 'drawer_handle' not in positions:
            raise KeyError("Expected key 'drawer_handle' in object_positions")
        drawer_handle_pos = positions['drawer_handle']
        zero_deg_quat = positions.get('zero_deg', [0, 0, 0, 1])
        ninety_deg_quat = positions.get('ninety_deg', [0, 0, 0, 1])
        
        # Step 1: Move close to the drawer handle
        print("[Task] Moving to drawer handle at:", drawer_handle_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=drawer_handle_pos,
                approach_distance=0.15,
                max_steps=200,
                threshold=0.01,
                approach_axis='z',
                timeout=15.0
            )
        except Exception as e:
            print(f"[Error] move to drawer handle failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely during move-to-handle.")
            return
        
        # Step 2: Ensure gripper is in zero_deg orientation before side approach
        print("[Task] Rotating gripper to zero_deg orientation")
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=zero_deg_quat,
                max_steps=150,
                threshold=0.03,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] rotate to zero_deg failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely during zero_deg rotation.")
            return
        
        # Step 3: Rotate gripper to ninety_deg to align for side movement
        print("[Task] Rotating gripper to ninety_deg orientation")
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=150,
                threshold=0.03,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] rotate to ninety_deg failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely during ninety_deg rotation.")
            return
        
        # Step 4: Pick the drawer handle
        print("[Task] Grasping drawer handle")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=drawer_handle_pos,
                approach_distance=0.02,
                max_steps=200,
                threshold=0.005,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] pick drawer handle failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely during pick.")
            return
        
        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer open")
        try:
            obs, reward, done = pull(
                env,
                task,
                target_pos=drawer_handle_pos,
                approach_distance=0.0,
                max_steps=200,
                threshold=0.005,
                approach_axis='x',
                timeout=12.0
            )
        except Exception as e:
            print(f"[Error] pull drawer failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely during pull.")
            return
        
        print("[Task] Drawer should now be open. Task complete.")
        
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()