import numpy as np
import time
from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, place, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all object positions in the scene
        positions = get_object_positions()
        print("[Info] Retrieved positions for objects:", list(positions.keys()))

        # === Exploration phase: visit each object to identify missing predicates ===
        identified = {}
        for name, pos in positions.items():
            print(f"[Exploration] Moving to '{name}' at {pos}")
            try:
                obs, reward, done = move(
                    env, task,
                    target_pos=pos,
                    approach_distance=0.1,
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )
            except Exception as e:
                print(f"[Exploration] Error moving to {name}: {e}")
                identified[name] = False
                continue
            if done:
                print("[Exploration] Task ended prematurely during exploration.")
                return
            # After move, the domain says we learn a predicate about objects in that location
            # We record that we successfully identified something at this location
            identified[name] = True
        print("[Exploration] Identification results:", identified)

        # === Oracle plan to open a drawer ===
        # We assume that object_positions supplies:
        #   'drawer_side_pos', 'drawer_anchor_pos', 'drawer_handle_pos'
        if not all(k in positions for k in ('drawer_side_pos', 'drawer_anchor_pos', 'drawer_handle_pos')):
            raise KeyError("Missing one of 'drawer_side_pos', 'drawer_anchor_pos', 'drawer_handle_pos' in positions")

        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']
        handle_pos = positions['drawer_handle_pos']

        # 1) Rotate gripper to the side orientation (ninety degrees)
        #    We assume ninety_deg quaternion is provided by positions or is known
        #    If not available, we define it here as an example.
        ninety_deg_quat = positions.get('ninety_deg_quat', np.array([0.0, 0.7071, 0.0, 0.7071]))
        print("[Plan] Rotating gripper to ninety degrees")
        try:
            obs, reward, done = rotate(
                env, task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        except Exception as e:
            print("[Plan] Error during rotate:", e)
            return
        if done:
            print("[Plan] Task ended during rotate.")
            return

        # 2) Move to the side of the drawer
        print(f"[Plan] Moving to side position {side_pos}")
        try:
            obs, reward, done = move(
                env, task,
                target_pos=side_pos,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print("[Plan] Error moving to side position:", e)
            return
        if done:
            print("[Plan] Task ended during side approach.")
            return

        # 3) Move to the anchor position (handle) of the drawer
        print(f"[Plan] Moving to anchor position {anchor_pos}")
        try:
            obs, reward, done = move(
                env, task,
                target_pos=anchor_pos,
                approach_distance=0.02,
                max_steps=100,
                threshold=0.005,
                timeout=10.0
            )
        except Exception as e:
            print("[Plan] Error moving to anchor position:", e)
            return
        if done:
            print("[Plan] Task ended during anchor approach.")
            return

        # 4) Pick (grasp) the drawer handle
        print(f"[Plan] Picking drawer handle at {handle_pos}")
        try:
            obs, reward, done = pick(
                env, task,
                target_pos=handle_pos,
                approach_distance=0.01,
                max_steps=50,
                threshold=0.005,
                timeout=5.0
            )
        except Exception as e:
            print("[Plan] Error during pick:", e)
            return
        if done:
            print("[Plan] Task ended during pick.")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling the drawer open")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print("[Plan] Error during pull:", e)
            return
        if done:
            print("[Plan] Task ended during pull.")
            return

        print("[Plan] Drawer should now be open!")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()