import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        
        # Initialize video recording if desired
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace 'object_1' and 'object_2' with actual keys from your positions dict
        object_1_pos = positions['object_1']
        object_2_pos = positions['object_2']
        
        # === Oracle Plan Execution ===
        # 1) Move robot to object_1's location (identification phase)
        print("[Task] Moving to object_1 at:", object_1_pos)
        try:
            obs, reward, done = move(env, task, from_location=None, to_location=object_1_pos)
        except Exception as e:
            print("[Error] move to object_1 failed:", e)
            return
        if done:
            print("[Task] Task ended after move to object_1!")
            return
        
        # 2) Pick up object_1
        print("[Task] Picking object_1 at:", object_1_pos)
        try:
            obs, reward, done = pick(env, task,
                                     obj='object_1',
                                     room=None,
                                     gripper=None)
        except Exception as e:
            print("[Error] pick object_1 failed:", e)
            return
        if done:
            print("[Task] Task ended after pick!")
            return
        
        # 3) Move robot to object_2's location
        print("[Task] Moving to object_2 at:", object_2_pos)
        try:
            obs, reward, done = move(env, task, from_location=None, to_location=object_2_pos)
        except Exception as e:
            print("[Error] move to object_2 failed:", e)
            return
        if done:
            print("[Task] Task ended after move to object_2!")
            return
        
        # 4) Place object_1 at object_2 location
        print("[Task] Placing object_1 at:", object_2_pos)
        try:
            obs, reward, done = place(env, task,
                                      obj='object_1',
                                      loc=object_2_pos)
        except Exception as e:
            print("[Error] place failed:", e)
            return
        if done:
            print("[Task] Task ended after place!")
            return
        
        # 5) (Optional) Rotate gripper if needed
        # target_quat should be a 4-element list [x, y, z, w]
        ninety_deg_quat = [0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety degrees")
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        except Exception as e:
            print("[Error] rotate failed:", e)
            return
        if done:
            print("[Task] Task ended after rotate!")
            return
        
        # 6) (Optional) Pull action if interacting with a drawer or similar
        print("[Task] Executing pull action")
        try:
            obs, reward, done = pull(env, task,
                                     obj=None,
                                     loc=None)
        except Exception as e:
            print("[Error] pull failed:", e)
            return
        if done:
            print("[Task] Task ended after pull!")
            return

    finally:
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()