import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve known object positions in the scene
        positions = get_object_positions()
        print("[Info] Available position keys:", list(positions.keys()))

        # you should replace these keys with actual ones from your scene
        drawer_side_pos = positions.get('drawer_side_pos')
        drawer_anchor_pos = positions.get('drawer_anchor_pos')
        trash_bin_pos = positions.get('trash_bin_pos')
        if drawer_side_pos is None or drawer_anchor_pos is None or trash_bin_pos is None:
            print("Error: required positions not found in scene description.")
            return

        # define a quaternion corresponding to a 90° rotation about the appropriate axis
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]

        # oracle plan: a sequence of high‑level skill calls and parameters
        oracle_plan = [
            {'action': 'rotate', 'params': {'target_quat': ninety_deg_quat}},
            {'action': 'move',   'params': {
                'target_pos': drawer_side_pos,
                'approach_distance': 0.10,
                'max_steps': 100,
                'threshold': 0.01,
                'approach_axis': 'z',
                'timeout': 10.0
            }},
            {'action': 'move',   'params': {
                'target_pos': drawer_anchor_pos,
                'approach_distance': 0.02,
                'max_steps': 100,
                'threshold': 0.005,
                'approach_axis': 'z',
                'timeout': 10.0
            }},
            {'action': 'pick',   'params': {
                'target_pos': drawer_anchor_pos,
                'approach_distance': 0.01,
                'max_steps': 50,
                'threshold': 0.005,
                'approach_axis': 'z',
                'timeout': 5.0
            }},
            {'action': 'pull',   'params': {}},
            {'action': 'move',   'params': {
                'target_pos': trash_bin_pos,
                'approach_distance': 0.15,
                'max_steps': 150,
                'threshold': 0.02,
                'approach_axis': 'z',
                'timeout': 15.0
            }},
            {'action': 'place',  'params': {
                'target_pos': trash_bin_pos,
                'approach_distance': 0.05,
                'max_steps': 100,
                'threshold': 0.01,
                'approach_axis': 'z',
                'timeout': 10.0
            }}
        ]

        # execute the oracle plan step by step
        for idx, step in enumerate(oracle_plan, start=1):
            action_name = step['action']
            params = step['params']
            print(f"[Plan] Step {idx}: {action_name} with params {params}")
            if action_name not in globals():
                print(f"  Warning: skill '{action_name}' is not defined, skipping.")
                continue
            skill_fn = globals()[action_name]
            try:
                obs, reward, done = skill_fn(env, task, **params)
            except Exception as e:
                print(f"  Error executing {action_name}: {e}")
                break
            if done:
                print(f"[Plan] Skill '{action_name}' returned done=True, terminating plan early.")
                break

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()