import numpy as np
import time
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve the known positions of objects in the scene
        positions = get_object_positions()
        print("[Task] Retrieved positions:", positions)

        # expect keys 'drawer_side_position' and 'drawer_anchor_position'
        drawer_side_pos = positions.get('drawer_side_position')
        drawer_anchor_pos = positions.get('drawer_anchor_position')
        if drawer_side_pos is None or drawer_anchor_pos is None:
            print("[Error] Drawer side or anchor position not found in positions dict.")
            return

        # 1) Rotate the gripper to a 90° orientation
        ninety_deg_quat = [0.0, 0.0, 0.7071, 0.7071]  # target quaternion for 90 degrees
        print("[Task] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_deg_quat,
                                   max_steps=100, threshold=0.03, timeout=10.0)
        if done:
            print("[Task] Finished during rotate step.")
            return

        # 2) Move gripper to side position of the drawer
        print(f"[Task] Moving to drawer side position: {drawer_side_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 approach_distance=0.15,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Finished during move-to-side step.")
            return

        # 3) Move gripper to anchor position (handle) of the drawer
        print(f"[Task] Moving to drawer anchor position: {drawer_anchor_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Finished during move-to-anchor step.")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Picking the drawer handle")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Finished during pick-drawer step.")
            return

        # 5) Pull open the drawer
        print("[Task] Pulling to open the drawer")
        obs, reward, done = pull(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.20,
                                 max_steps=100,
                                 threshold=0.02,
                                 approach_axis='x',
                                 timeout=5.0)
        if done:
            print("[Task] Finished during pull step.")
            return

        print("[Task] Plan execution completed successfully.")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()