import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import pick, place, rotate, pull, move
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation so we record everything
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve object positions ===
        positions = get_object_positions()
        # We expect positions to contain keys for our objects and a disposal bin
        # Example keys: 'apple', 'banana', 'disposal_bin'
        if 'disposal_bin' not in positions:
            raise KeyError("Expected a key 'disposal_bin' in object_positions() output")
        disposal_pos = positions['disposal_bin']
        
        # === Example: pick every object (except the disposal bin) and place it in the disposal bin ===
        for name, pos in positions.items():
            if name == 'disposal_bin':
                continue
            print(f"[Task] Manipulating '{name}' at position {pos}")
            try:
                # Approach and pick the object
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Early termination after pick of {name}.")
                    return

                # Place the object into the disposal bin
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=disposal_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Early termination after place of {name}.")
                    return

            except Exception as e:
                print(f"[Task] Error manipulating '{name}': {e}")
                # continue to next object after logging
                continue

        # === (Optionally) open a drawer if one exists in positions ===
        # If our environment has a drawer handle location, we can rotate, pick-drawer, then pull
        if 'drawer_side_pos' in positions and 'drawer_anchor_pos' in positions:
            side = positions['drawer_side_pos']
            anchor = positions['drawer_anchor_pos']
            # define target quaternion for 90-degree rotation around gripper axis
            ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
            zero_quat = obs.gripper_pose[3:7]
            # rotate to side view
            obs, reward, done = rotate(env, task, ninety_deg_quat)
            # move along side to anchor (assuming move takes (env,task,from_pos,to_pos,...))
            obs, reward, done = move(env, task, side, anchor, max_steps=100, threshold=0.01, timeout=10.0)
            # pick the drawer handle (this skill is named pick in our import)
            obs, reward, done = pick(
                env, task,
                target_pos=anchor,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=5.0
            )
            # pull to open / identify lock state
            obs, reward, done = pull(env, task)
            # return gripper to original orientation
            obs, reward, done = rotate(env, task, zero_quat)
            if done:
                print("[Task] Task ended during drawer operation.")
        
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()