import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve object positions ===
        positions = get_object_positions()
        # Replace these keys with actual names in your environment
        # positions should contain:
        #   'drawer_side_position'
        #   'drawer_anchor_position'
        #   'drawer_handle'
        #   (optionally other objects for downstream tasks)
        drawer_side_pos     = positions['drawer_side_position']
        drawer_anchor_pos   = positions['drawer_anchor_position']
        drawer_handle_pos   = positions['drawer_handle']

        # === Oracle plan to open a drawer ===

        # 1) Rotate gripper to 90 degrees
        #    quaternion for 90° about z‐axis: [x, y, z, w]
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Task] Rotating gripper to 90 degrees")
        try:
            obs, reward, done = rotate(
                env, task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.02,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Task] Task ended during rotate")
            return

        # 2) Move to side position of drawer
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        try:
            obs, reward, done = move(
                env, task,
                target_pos=drawer_side_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] move-to-side failed: {e}")
            return
        if done:
            print("[Task] Task ended during move-to-side")
            return

        # 3) Move to anchor (handle) position of drawer
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        try:
            obs, reward, done = move(
                env, task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] move-to-anchor failed: {e}")
            return
        if done:
            print("[Task] Task ended during move-to-anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking drawer handle at:", drawer_handle_pos)
        try:
            obs, reward, done = pick(
                env, task,
                target_pos=drawer_handle_pos,
                approach_distance=0.04,
                max_steps=100,
                threshold=0.005,
                approach_axis='x',
                timeout=5.0
            )
        except Exception as e:
            print(f"[Error] pick-handle failed: {e}")
            return
        if done:
            print("[Task] Task ended during pick-handle")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Task ended during pull")
            return

        print("[Task] Drawer opened successfully!")

        # === (Optional) continue with picking items from the drawer ===
        # Example:
        # item_pos = positions['target_item']
        # print("[Task] Picking item from opened drawer at:", item_pos)
        # obs, reward, done = pick(env, task, target_pos=item_pos, approach_distance=0.08, ...)
        # if done: return
        # place_pos = positions['place_location']
        # print("[Task] Placing item at:", place_pos)
        # obs, reward, done = place(env, task, target_pos=place_pos, ...)
        # if done: return

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
