import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull, pick_drawer

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions from the helper module
        positions = get_object_positions()
        # Expect keys like 'drawer_side_positions' and 'drawer_anchor_positions'
        if 'drawer_side_positions' not in positions or 'drawer_anchor_positions' not in positions:
            raise KeyError("Missing required position keys: 'drawer_side_positions' or 'drawer_anchor_positions'")

        # Pick the first candidate side and anchor positions
        side_pos = np.array(positions['drawer_side_positions'][0])
        anchor_pos = np.array(positions['drawer_anchor_positions'][0])

        # 1) Move gripper to a side position of the drawer
        print("[Task] Moving to drawer side position:", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended prematurely after moving to side.")
            return

        # 2) Rotate gripper so it is aligned for the handle pull
        #    We assume a 'rotate' skill that takes (env, task, from_angle, to_angle)
        #    You may need to adapt the angle names ('zero_deg','ninety_deg') to your setup
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended prematurely after rotation.")
            return

        # 3) Move gripper to the anchor (handle) position
        print("[Task] Moving to drawer anchor position:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended prematurely after moving to anchor.")
            return

        # 4) Engage the pick_drawer skill to grasp the handle
        #    Assume signature: pick_drawer(env, task, drawer_name, gripper_name, position)
        #    Here we leave drawer_name and gripper_name as defaults or infer from descriptions
        print("[Task] Picking drawer handle")
        obs, reward, done = pick_drawer(env, task)
        if done:
            print("[Task] Task ended after picking drawer handle.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended after pulling.")
            return

        print("[Task] Drawer should now be open. Task complete!")
    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()