import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        # positions is a dict mapping object keys to 3D positions (and possibly quaternions)
        positions = get_object_positions()

        # Make sure the keys match your objects in the scene.
        # Here we assume two objects named 'object_1' and 'object_2' in positions.
        if 'object_1' not in positions or 'object_2' not in positions:
            raise KeyError("Expected keys 'object_1' and 'object_2' in positions")

        object_1_pos = positions['object_1']
        object_2_pos = positions['object_2']

        # === Oracle Plan Execution ===

        # 1) Approach and pick up object_1
        print("[Task] Picking object_1 at:", object_1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=object_1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick()")
            return

        # 2) Place object_1 at object_2's location
        print("[Task] Placing object_1 at object_2 location:", object_2_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=object_2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during place()")
            return

        # 3) (Optional) If there were further steps, e.g., rotating a handle or pulling:
        #    target_quat = positions.get('drawer_handle_quat', None)
        #    if target_quat is not None:
        #        print("[Task] Rotating gripper to open drawer")
        #        obs, reward, done = rotate(env, task, target_quat, max_steps=100, threshold=0.05, timeout=10.0)
        #        if done:
        #            print("[Task] Task ended during rotate()")
        #            return
        #
        #    # after rotate, you might call pull(env, task) etc.
        #    # obs, reward, done = pull(env, task)
        #    # if done:
        #    #     print("[Task] Task ended during pull()")
        #    #     return

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()