import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Assuming object_positions returns keys 'drawer_side_pos' and 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected keys 'drawer_side_pos' and 'drawer_anchor_pos' in positions dict")
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # === Oracle Plan Execution ===

        # 1) Rotate gripper to 90 degrees around its local axis
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to 90° orientation")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Finished during rotation step.")
            return

        # 2) Move gripper to the side position of the drawer
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Finished during move-to-side step.")
            return

        # 3) Move gripper to the anchor (handle) position of the drawer
        print("[Task] Moving to drawer handle (anchor) position:", drawer_anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='x',
                                 timeout=10.0)
        if done:
            print("[Task] Finished during move-to-anchor step.")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.01,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='x',
                                 timeout=10.0)
        if done:
            print("[Task] Finished during pick-drawer step.")
            return

        # 5) Pull the drawer to open it
        print("[Task] Pulling the drawer open")
        # If pull takes additional parameters, adjust accordingly
        try:
            obs, reward, done = pull(env, task)
        except TypeError:
            # fallback if pull requires explicit distance and timeout
            obs, reward, done = pull(env, task,
                                     pull_distance=0.2,
                                     max_steps=100,
                                     timeout=10.0)
        if done:
            print("[Task] Finished during pull step.")
            return

        print("[Task] Oracle plan completed successfully.")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()