import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import rotate, move, pick, place, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all object positions in the scene
        positions = get_object_positions()
        # exploration phase: try moving to each object to identify missing predicate
        print("[Exploration] Starting identification of object properties")
        # assume robot starts at the first location in positions
        loc_keys = list(positions.keys())
        if not loc_keys:
            print("[Exploration] No objects found, skipping exploration")
        else:
            current_loc = loc_keys[0]
            for obj_name, obj_pos in positions.items():
                print(f"[Exploration] Moving from '{current_loc}' to '{obj_name}' at {obj_pos}")
                try:
                    obs, reward, done = move(env, task, current_loc, obj_name)
                except Exception as e:
                    print(f"[Exploration] Error during move to {obj_name}: {e}")
                    break
                current_loc = obj_name
                if done:
                    print("[Exploration] Task ended during exploration")
                    return
            print("[Exploration] Completed movement exploration. Next: pick up objects to learn weight/durability")
            # pick each object once to learn weight or durability
            for obj_name in positions.keys():
                print(f"[Exploration] Attempting pick on '{obj_name}' at its location")
                try:
                    obs, reward, done = pick(env, task, obj_name, obj_name)
                except Exception as e:
                    print(f"[Exploration] Error during pick of {obj_name}: {e}")
                    continue
                if done:
                    print("[Exploration] Task ended after pick")
                    return
            print("[Exploration] Exploration phase done. Proceeding to drawer opening plan")

        # now execute the drawer-opening plan
        # define the drawer identifiers and poses
        drawer_name = 'drawer1'
        side_pos_key = drawer_name + '_side'
        anchor_pos_key = drawer_name + '_anchor'
        if side_pos_key not in positions or anchor_pos_key not in positions:
            print(f"[Plan] Missing side/anchor positions for {drawer_name}, aborting plan")
            return
        # 1) rotate gripper from zero to ninety degrees
        ninety_deg_quat = [0.0, 0.0, 0.7071, 0.7071]
        print(f"[Plan] Rotating gripper to ninety degrees quaternion {ninety_deg_quat}")
        try:
            obs, reward, done = rotate(env, task, ninety_deg_quat)
        except Exception as e:
            print(f"[Plan] Error during rotate: {e}")
            return
        if done:
            print("[Plan] Task ended during rotate")
            return

        # 2) move from current anchor to side position
        print(f"[Plan] Moving gripper to side position '{side_pos_key}'")
        try:
            obs, reward, done = move(env, task, anchor_pos_key, side_pos_key)
        except Exception as e:
            print(f"[Plan] Error during move-to-side: {e}")
            return
        if done:
            print("[Plan] Task ended during move-to-side")
            return

        # 3) move from side position back to anchor
        print(f"[Plan] Moving gripper back to anchor position '{anchor_pos_key}'")
        try:
            obs, reward, done = move(env, task, side_pos_key, anchor_pos_key)
        except Exception as e:
            print(f"[Plan] Error during move-to-anchor: {e}")
            return
        if done:
            print("[Plan] Task ended during move-to-anchor")
            return

        # 4) pick the drawer handle
        print(f"[Plan] Picking drawer handle on '{drawer_name}' at anchor position")
        try:
            obs, reward, done = pick(env, task, drawer_name, anchor_pos_key)
        except Exception as e:
            print(f"[Plan] Error during pick-drawer: {e}")
            return
        if done:
            print("[Plan] Task ended during pick-drawer")
            return

        # 5) pull to open the drawer
        print(f"[Plan] Pulling '{drawer_name}' to open it")
        try:
            obs, reward, done = pull(env, task, drawer_name, anchor_pos_key)
        except Exception as e:
            print(f"[Plan] Error during pull: {e}")
            return
        if done:
            print("[Plan] Task ended after pull")
            return

        print("[Plan] Drawer opening sequence complete")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()