import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task in simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # The keys below must match entries in your object_positions module.
        # Try common naming conventions, fall back if missing.
        try:
            side_pos = positions['bottom_side_pos']
        except KeyError:
            side_pos = positions.get('drawer_side_pos')
        try:
            anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            anchor_pos = positions.get('drawer_anchor_pos')

        if side_pos is None or anchor_pos is None:
            raise RuntimeError("Could not find required positions for drawer handle.")

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper from its default angle to ninety degrees
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task,
                                   gripper_name='gripper',
                                   from_angle='zero_deg',
                                   to_angle='ninety_deg')
        if done:
            print("[Task] Episode ended during rotate.")
            return

        # 2) Move gripper to the side of the drawer
        print(f"[Task] Moving to side position: {side_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=side_pos,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=5.0)
        if done:
            print("[Task] Episode ended during move-to-side.")
            return

        # 3) Move gripper to the anchor position (handle)
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=anchor_pos,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=5.0)
        if done:
            print("[Task] Episode ended during move-to-anchor.")
            return

        # 4) Pick the drawer handle
        print("[Task] Gripping handle")
        obs, reward, done = pick(env, task,
                                 target_pos=anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Task] Episode ended during pick-drawer.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended during pull.")
            return

        print("===== Task completed: Drawer is now open =====")

    except Exception as e:
        print(f"[Error] {type(e).__name__}: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
